<?php

namespace App\Filament\Resources\KategoriResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Notifications\Notification;
use Filament\Support\Exceptions\Halt;

class YemeklerRelationManager extends RelationManager
{
    protected static string $relationship = 'yemekler';

    protected static ?string $recordTitleAttribute = 'name';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Card::make()
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->required()
                            ->label('Yemek Adı')
                            ->placeholder('Yemek adını giriniz')
                            ->maxLength(255)
                            ->helperText(fn ($record) => $record?->description ?? '')
                            ->columnSpan('full'),

                        Forms\Components\FileUpload::make('image')
                            ->image()
                            ->directory('meal-images')
                            ->label('Görsel')
                            ->imagePreviewHeight('150')
                            ->placeholder('Yemek görselini yükleyiniz')
                            ->columnSpan('full')
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('16:9')
                            ->imageResizeTargetWidth('1920')
                            ->imageResizeTargetHeight('1080'),

                        Forms\Components\Textarea::make('description')
                            ->label('Açıklama')
                            ->rows(4)
                            ->placeholder('Yemek açıklaması...'),

                        Forms\Components\TextInput::make('price')
                            ->numeric()
                            ->label('Fiyat')
                            ->placeholder('Fiyatı giriniz')
                            ->columnSpan('full'),

                        Forms\Components\Toggle::make('is_active')
                            ->label('Aktif mi?')
                            ->inline(false)
                            ->onColor('success')
                            ->offColor('danger')
                            ->helperText('Yemeğin aktif/pasif durumunu ayarlayın.')
                            ->columnSpan('full'),

                        Forms\Components\Toggle::make('vitrin')
                            ->label('Kategori Vitrini')
                            ->inline(false)
                            ->onColor('success')
                            ->offColor('danger')
                            ->helperText('Bu yemeğin kategori vitrinde görünmesini sağlar.')
                            ->columnSpan('full'),
                    ]),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->defaultSort('order')
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Yemek Adı')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\ImageColumn::make('image')
                    ->label('Görsel')
                    ->circular(),

                Tables\Columns\TextColumn::make('price')
                    ->label('Fiyat')
                    ->money('TRY')
                    ->sortable(),

                Tables\Columns\IconColumn::make('is_active')
                    ->boolean()
                    ->label('Durum')
                    ->sortable(),

                Tables\Columns\IconColumn::make('vitrin')
                    ->label('Kategori Vitrini')
                    ->boolean()
                    ->colors([
                        'success' => fn ($state) => $state === true,
                        'danger' => fn ($state) => $state === false,
                    ]),
            ])
            ->filters([])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->before(function () {
                        if (config('app.demo')) {
                            Notification::make()
                                ->title('Demo Mod')
                                ->body('Ekleme işlemi demo modunda devre dışıdır.')
                                ->danger()
                                ->send();

                            throw new Halt();
                        }
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->before(function () {
                        if (config('app.demo')) {
                            Notification::make()
                                ->title('Demo Mod')
                                ->body('Düzenleme işlemi demo modunda devre dışıdır.')
                                ->danger()
                                ->send();

                            throw new Halt();
                        }
                    }),
                Tables\Actions\DeleteAction::make()
                    ->before(function () {
                        if (config('app.demo')) {
                            Notification::make()
                                ->title('Demo Mod')
                                ->body('Silme işlemi demo modunda devre dışıdır.')
                                ->danger()
                                ->send();

                            throw new Halt();
                        }
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->before(function () {
                            if (config('app.demo')) {
                                Notification::make()
                                    ->title('Demo Mod')
                                    ->body('Toplu silme işlemi demo modunda devre dışıdır.')
                                    ->danger()
                                    ->send();

                                throw new Halt();
                            }
                        }),
                ]),
            ]);
    }
}
