<?php

namespace App\Filament\Resources;

use App\Filament\Resources\KategoriResource\Pages;
use App\Filament\Resources\KategoriResource\RelationManagers;
use App\Models\Kategori;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\Reorderable;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Support\Exceptions\Halt;
use Filament\Notifications\Notification;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Card;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class KategoriResource extends Resource
{
    protected static ?string $model = Kategori::class;

    protected static ?string $navigationGroup = 'Ana Sayfa';
    protected static ?string $navigationLabel = 'Kategoriler';
    protected static ?string $pluralModelLabel = 'Kategoriler';
    protected static ?int $navigationSort = 2;
    protected static ?string $slug = 'kategoriler';
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Card::make()
                    ->schema([
                        TextInput::make('name')
                            ->required()
                            ->label('Kategori Adı')
                            ->placeholder('Kategori adını buraya yazınız')
                            ->helperText('Kategori adı en az 3 karakter olmalıdır.')
                            ->minLength(3)
                            ->maxLength(255)
                            ->columnSpan('full'),

                        FileUpload::make('image')
                            ->image()
                            ->directory('kategori-gorselleri')
                            ->label('Kategori Görseli')
                            ->imagePreviewHeight('150')
                            ->placeholder('Kategori görselini buraya yükleyiniz')
                            ->columnSpan('full')
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('16:9')
                            ->imageResizeTargetWidth('1920')
                            ->imageResizeTargetHeight('1080'),

                        Toggle::make('is_active')
                            ->label('Aktif Mi?')
                            ->inline(false)
                            ->onColor('success')
                            ->offColor('danger')
                            ->helperText('Kategoriyi aktif veya pasif yapabilirsiniz.')
                            ->columnSpan('full'),
                    ])
                    ->columns(1),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->defaultSort('order')
            ->columns([
                TextColumn::make('name')
                    ->label('Kategori Adı')
                    ->searchable()
                    ->sortable()
                    ->toggleable()
                    ->extraAttributes(['class' => 'font-semibold text-lg']),

                TextColumn::make('yemekler_count')
                    ->label('Yemek Sayısı')
                    ->counts('yemekler')
                    ->sortable()
                    ->color('primary')
                    ->tooltip(fn($record) => $record->yemekler_count . ' yemek mevcut'),

                ImageColumn::make('image')
                    ->label('Görsel')
                    ->circular()
                    ->size(48)
                    ->toggleable(),

                IconColumn::make('is_active')
                    ->boolean()
                    ->label('Aktiflik')
                    ->sortable()
                    ->colors([
                        'success' => fn($state) => $state === true,
                        'danger' => fn($state) => $state === false,
                    ])
                    ->toggleable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->before(function () {
                        if (config('app.demo', false)) {
                            Notification::make()
                                ->danger()
                                ->title('Demo modunda bu işlem yapılamaz.')
                                ->send();

                            throw new Halt;
                        }
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->before(function () {
                        if (config('app.demo', false)) {
                            Notification::make()
                                ->danger()
                                ->title('Demo modunda bu işlem yapılamaz.')
                                ->send();

                            throw new Halt;
                        }
                    }),
                ]),
            ])
            ->emptyStateHeading('Henüz kategori bulunmamaktadır.')
            ->emptyStateDescription('Kategori eklemek için yukarıdaki butonu kullanabilirsiniz.')
            ->emptyStateIcon('heroicon-o-folder-open');
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\YemeklerRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListKategoris::route('/'),
            'create' => Pages\CreateKategori::route('/create'),
            'edit' => Pages\EditKategori::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->withCount('yemekler');
    }
}
