/**
 * API Response Types
 * SOLID Principle: SRP - API contract definitions
 */

/**
 * Generic API Response wrapper
 */
export interface ApiResponse<T> {
  data: T
  success: boolean
  message?: string
  timestamp?: number
}

/**
 * Paginated response
 */
export interface PaginatedResponse<T> {
  data: T[]
  pagination: {
    page: number
    limit: number
    total: number
    totalPages: number
    hasNext: boolean
    hasPrev: boolean
  }
  success: boolean
  message?: string
}

/**
 * Error response
 */
export interface ApiError {
  success: false
  error: {
    code: string
    message: string
    details?: Record<string, string[]>
  }
  timestamp: number
}

/**
 * User related types
 */
export interface UserProfile {
  id: string
  name: string
  username: string
  email: string
  phone: string
  avatar?: string
  isEmailVerified: boolean
  isPhoneVerified: boolean
  referralCode: string
  joinDate: string
  balance: number
  stats?: {
    totalEarned: number
    totalTasks: number
    totalReferrals: number
  }
}

export interface UpdateProfileRequest {
  name?: string
  email?: string
  phone?: string
  avatar?: string
}

export interface ChangePasswordRequest {
  currentPassword: string
  newPassword: string
  confirmPassword: string
}

/**
 * Verification related types
 */
export interface SendVerificationCodeRequest {
  target: string // email or phone
  type: 'email' | 'phone'
}

export interface VerifyCodeRequest {
  target: string
  code: string
  type: 'email' | 'phone'
}

export interface VerificationResponse {
  verified: boolean
  message: string
}

/**
 * Withdrawal related types
 */
export interface BankWithdrawalRequest {
  bankId: string
  accountName: string
  iban: string
  amount: number
}

export interface CryptoWithdrawalRequest {
  walletAddress: string
  amount: number
  currency: 'USDC' | 'USDT'
}

export interface WithdrawalResponse {
  id: string
  status: 'pending' | 'processing' | 'completed' | 'rejected'
  amount: number
  fee: number
  netAmount: number
  createdAt: string
  estimatedCompletion?: string
}

/**
 * Task related types
 */
export interface TaskResponse {
  id: string
  title: string
  description: string
  type: 'micro' | 'macro'
  category: string
  amount: number
  status: 'available' | 'in-progress' | 'completed' | 'expired'
  deadline?: string
  requirements?: string[]
  steps?: TaskStep[]
  partner?: {
    name: string
    logo: string
    rating?: number
  }
}

export interface TaskStep {
  id: string
  order: number
  title: string
  description: string
  isCompleted: boolean
}

export interface StartTaskRequest {
  taskId: string
}

export interface CompleteTaskRequest {
  taskId: string
  proof?: string
  answers?: Record<string, string>
}

/**
 * Leaderboard related types
 */
export interface LeaderboardEntry {
  rank: number
  userId: string
  username: string
  avatar?: string
  score: number
  totalEarned: number
  isCurrentUser?: boolean
}

export interface LeaderboardResponse {
  entries: LeaderboardEntry[]
  currentUserRank?: number
  period: 'daily' | 'weekly' | 'monthly' | 'all-time'
}

/**
 * Support related types
 */
export interface SupportTicket {
  id: string
  subject: string
  category: string
  status: 'open' | 'in-progress' | 'resolved' | 'closed'
  priority: 'low' | 'medium' | 'high'
  createdAt: string
  updatedAt: string
  messages: SupportMessage[]
}

export interface SupportMessage {
  id: string
  ticketId: string
  sender: 'user' | 'support'
  message: string
  createdAt: string
  attachments?: string[]
}

export interface CreateSupportTicketRequest {
  subject: string
  category: string
  message: string
  priority?: 'low' | 'medium' | 'high'
}

/**
 * Notification related types
 */
export interface Notification {
  id: string
  type: 'info' | 'success' | 'warning' | 'error'
  title: string
  message: string
  read: boolean
  createdAt: string
  link?: string
  icon?: string
}

export interface NotificationPreferences {
  email: boolean
  sms: boolean
  push: boolean
  taskUpdates: boolean
  withdrawalUpdates: boolean
  referralUpdates: boolean
}
