/**
 * usePagination Hook
 * Pagination mantığını yöneten reusable hook
 * Backend ile entegrasyon için hazır
 */

import { useState, useMemo } from 'react'

export interface UsePaginationProps<T> {
  /** Tüm veri array'i */
  data: T[]
  /** Sayfa başına gösterilecek öğe sayısı */
  itemsPerPage?: number
  /** Başlangıç sayfası */
  initialPage?: number
}

export interface UsePaginationReturn<T> {
  /** Mevcut sayfadaki veriler */
  currentData: T[]
  /** Mevcut sayfa numarası */
  currentPage: number
  /** Toplam sayfa sayısı */
  totalPages: number
  /** Toplam öğe sayısı */
  totalItems: number
  /** Belirli bir sayfaya git */
  goToPage: (page: number) => void
  /** Sonraki sayfaya git */
  nextPage: () => void
  /** Önceki sayfaya git */
  previousPage: () => void
  /** İlk sayfaya git */
  firstPage: () => void
  /** Son sayfaya git */
  lastPage: () => void
  /** İlk sayfada mı? */
  isFirstPage: boolean
  /** Son sayfada mı? */
  isLastPage: boolean
  /** Mevcut sayfa aralığı (örn: "1-10 / 100") */
  pageRange: string
}

export function usePagination<T>({
  data,
  itemsPerPage = 21, // 3x7 grid için optimize
  initialPage = 1
}: UsePaginationProps<T>): UsePaginationReturn<T> {
  const [currentPage, setCurrentPage] = useState(initialPage)

  // Toplam sayfa sayısını hesapla
  const totalPages = useMemo(() => {
    return Math.ceil(data.length / itemsPerPage)
  }, [data.length, itemsPerPage])

  // Mevcut sayfadaki verileri hesapla
  const currentData = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage
    const endIndex = startIndex + itemsPerPage
    return data.slice(startIndex, endIndex)
  }, [data, currentPage, itemsPerPage])

  // Sayfa değiştirme fonksiyonları
  const goToPage = (page: number) => {
    const pageNumber = Math.max(1, Math.min(page, totalPages))
    setCurrentPage(pageNumber)
  }

  const nextPage = () => {
    if (currentPage < totalPages) {
      setCurrentPage(prev => prev + 1)
    }
  }

  const previousPage = () => {
    if (currentPage > 1) {
      setCurrentPage(prev => prev - 1)
    }
  }

  const firstPage = () => {
    setCurrentPage(1)
  }

  const lastPage = () => {
    setCurrentPage(totalPages)
  }

  // Durum kontrolleri
  const isFirstPage = currentPage === 1
  const isLastPage = currentPage === totalPages || totalPages === 0

  // Sayfa aralığı string'i
  const pageRange = useMemo(() => {
    const start = (currentPage - 1) * itemsPerPage + 1
    const end = Math.min(currentPage * itemsPerPage, data.length)
    return `${start}-${end} / ${data.length}`
  }, [currentPage, itemsPerPage, data.length])

  return {
    currentData,
    currentPage,
    totalPages,
    totalItems: data.length,
    goToPage,
    nextPage,
    previousPage,
    firstPage,
    lastPage,
    isFirstPage,
    isLastPage,
    pageRange
  }
}

export default usePagination
