/**
 * Featured Tasks Data
 * Separation of Concerns: Data layer
 * Single Responsibility: Task data generation and management
 */

export interface FeaturedTaskCardData {
  id: string
  amount: string
  title: string
  userImage: string
  userName: string
  userNickname: string
}

// Task types with Turkish titles
const taskTypes = [
  'Çekim',
  'Paylaşım',
  'İnceleme',
  'Video',
  'Reklam',
  'Beğeni',
  'Yorum',
  'Takip',
  'Story',
  'Anket'
]

// Turkish names for realistic data
const userNames = [
  'Ahmet Yılmaz',
  'Mehmet Kaya',
  'Ayşe Demir',
  'Fatma Şahin',
  'Zeynep Çelik',
  'Ali Öztürk',
  'Hüseyin Aydın',
  'Emine Arslan',
  'Mustafa Yıldız',
  'Hatice Doğan',
  'İbrahim Aslan',
  'Elif Polat',
  'Mahmut Şen',
  'Meryem Kurt',
  'Osman Özkan'
]

// User avatars
const avatars = [
  '/assets/images/pp1.png',
  '/assets/images/pp2.png'
]

/**
 * Generates a random featured task
 * @returns {FeaturedTaskCardData} A randomly generated task
 */
export const generateFeaturedTask = (): FeaturedTaskCardData => {
  const randomName = userNames[Math.floor(Math.random() * userNames.length)]
  const randomTaskType = taskTypes[Math.floor(Math.random() * taskTypes.length)]
  const randomAmount = (Math.random() * (6000 - 1000) + 1000).toFixed(0)
  const randomAvatar = avatars[Math.floor(Math.random() * avatars.length)]
  
  // Generate nickname from name
  const nickname = randomName
    .toLowerCase()
    .replace(/ı/g, 'i')
    .replace(/ğ/g, 'g')
    .replace(/ü/g, 'u')
    .replace(/ş/g, 's')
    .replace(/ö/g, 'o')
    .replace(/ç/g, 'c')
    .replace(/\s+/g, '')

  return {
    id: `task-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
    amount: `${randomAmount}₺`,
    title: randomTaskType,
    userImage: randomAvatar,
    userName: randomName,
    userNickname: `@${nickname}`
  }
}

/**
 * Gets initial featured tasks
 * @param {number} count - Number of tasks to generate (default: 10)
 * @returns {FeaturedTaskCardData[]} Array of featured tasks
 */
export const getInitialFeaturedTasks = (count: number = 10): FeaturedTaskCardData[] => {
  return Array.from({ length: count }, () => generateFeaturedTask())
}

// Export initial tasks for backward compatibility
export const featuredTasks: FeaturedTaskCardData[] = getInitialFeaturedTasks(10)
