'use client'

import React from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { cn } from '@/lib/utils'
import { Iconify } from './Iconify'

interface ModalProps {
  isOpen: boolean
  onClose: () => void
  children: React.ReactNode
  size?: 'sm' | 'md' | 'lg' | 'xl'
  showCloseButton?: boolean
  className?: string
  image?: string
  mobileImage?: string
  imageAlt?: string
  fullscreen?: boolean
}

export function Modal({
  isOpen,
  onClose,
  children,
  size = 'md',
  showCloseButton = true,
  className,
  image,
  mobileImage,
  imageAlt,
  fullscreen = false
}: ModalProps) {
  // ESC tuşu ile kapatma
  React.useEffect(() => {
    const handleEscape = (e: KeyboardEvent) => {
      if (e.key === 'Escape' && isOpen) {
        onClose()
      }
    }

    if (isOpen) {
      document.addEventListener('keydown', handleEscape)
      document.body.style.overflow = 'hidden'
    }

    return () => {
      document.removeEventListener('keydown', handleEscape)
      document.body.style.overflow = 'unset'
    }
  }, [isOpen, onClose])

  const sizeClasses = {
    sm: 'max-w-md',
    md: image ? 'max-w-full md:max-w-[1000px]' : 'max-w-md',
    lg: 'max-w-4xl',
    xl: 'max-w-5xl'
  }

  // Animation variants
  const backdropVariants = {
    hidden: { opacity: 0 },
    visible: { opacity: 1 }
  }

  const modalVariants = {
    hidden: { 
      opacity: 0,
      scale: 0.95,
      y: 20
    },
    visible: { 
      opacity: 1,
      scale: 1,
      y: 0,
      transition: {
        type: "spring" as const,
        stiffness: 300,
        damping: 30
      }
    },
    exit: {
      opacity: 0,
      scale: 0.95,
      y: 20,
      transition: {
        duration: 0.2
      }
    }
  }

  return (
    <AnimatePresence mode="wait">
      {isOpen && (
        <div className="fixed inset-0 z-60 overflow-y-auto max-h-screen">
          {/* Backdrop */}
          <motion.div 
            className="fixed inset-0 bg-black/60 backdrop-blur-sm"
            onClick={onClose}
            variants={backdropVariants}
            initial="hidden"
            animate="visible"
            exit="hidden"
            transition={{ duration: 0.2 }}
          />
          
          {/* Modal Container */}
          <div className={cn(
            "flex items-center justify-center",
            fullscreen ? "min-h-screen md:min-h-full min-w-screen md:w-fit md:p-4" : "min-h-full p-4 md:p-0"
          )}>
            <motion.div 
              className={cn(
                'relative w-full bg-base-1 border border-primary/20 shadow-2xl overflow-hidden',
                fullscreen 
                  ? 'inset-0 md:rounded-3xl h-screen md:h-auto' 
                  : 'rounded-2xl md:rounded-3xl mx-4 md:mx-0 my-4 md:my-0',
                !fullscreen && sizeClasses[size],
                fullscreen && 'md:max-w-md',
                fullscreen && image && 'md:max-w-[1000px]',
                className
              )}
              onClick={(e) => e.stopPropagation()}
              variants={modalVariants}
              initial="hidden"
              animate="visible"
              exit="exit"
            >
              {/* Close Button */}
              {showCloseButton && (
                <motion.button
                  onClick={onClose}
                  className="absolute top-3 right-3 md:top-4 md:right-4 z-10 text-gray-400 flex items-center justify-center hover:text-white transition-colors p-2 rounded-lg hover:bg-white/3 bg-black/20 md:bg-transparent"
                  whileHover={{ scale: 1.1, rotate: 90 }}
                  whileTap={{ scale: 0.9 }}
                  transition={{ duration: 0.2 }}
                >
                  <Iconify icon="material-symbols:close" size={20} className="md:text-[25px]" />
                </motion.button>
              )}

              <div className="flex flex-col md:flex-row h-full">
                {/* Top Side (Mobile) / Left Side (Desktop) - Image (Optional) */}
                {image && (
                  <motion.div 
                    className="h-[280px] md:h-full w-full md:w-[45%] md:min-w-[45%] rounded-t-2xl md:rounded-l-3xl md:rounded-tr-none overflow-hidden relative"
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ delay: 0.1, duration: 0.3 }}
                  >
                    {/* Mobile Image */}
                    {mobileImage && (
                      <img
                        src={mobileImage}
                        alt={imageAlt || 'Modal image'}
                        className="md:hidden w-full h-full object-cover object-center"
                      />
                    )}
                    {/* Desktop Image */}
                    <img
                      src={image}
                      alt={imageAlt || 'Modal image'}
                      className={cn(
                        "w-full h-full object-cover object-center",
                        mobileImage ? "hidden md:block" : ""
                      )}
                    />
                    {/* Gradient overlay for better text readability if needed */}
                    <div className="absolute inset-0 bg-gradient-to-b md:bg-gradient-to-r from-transparent to-base-1/20" />
                  </motion.div>
                )}
                
                {/* Bottom Side (Mobile) / Right Side (Desktop) - Content */}
                <motion.div 
                  className={cn(
                    'flex-1 p-4 md:p-6  scrollbar-hide overflow-y-auto',
                    image ? '' : 'h-full',
                    image ? '' : 'w-full'
                  )}
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  transition={{ delay: 0.15, duration: 0.3 }}
                >
                  {children}
                </motion.div>
              </div>
            </motion.div>
          </div>
        </div>
      )}
    </AnimatePresence>
  )
}