'use client'

import { useState } from 'react'
import { motion } from 'framer-motion'
import { Modal, Button, Icon, Iconify } from '@/components/ui'
import Image from 'next/image'
import type { OfferCardData, TaskReward } from '@/data/offers'

interface TaskDetailModalProps {
  isOpen: boolean
  onClose: () => void
  task: OfferCardData | null
}

export function TaskDetailModal({ isOpen, onClose, task }: TaskDetailModalProps) {
  const [activeTab, setActiveTab] = useState<'oduller' | 'detay'>('oduller')

  if (!task) return null

  return (
    <Modal isOpen={isOpen} onClose={onClose} fullscreen size="md" showCloseButton>
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3 }}
      >
        <h2 className="text-white text-lg font-bold mb-2">{task.title}</h2>
        
        <div className="flex flex-col items-center mb-4">
          {/* Görev Görseli */}
          <div className="w-full flex justify-center mb-4">
            <Image 
              src={task.image} 
              alt={task.title} 
              width={500} 
              height={500} 
              unoptimized
              className="rounded-xl object-cover" 
            />
          </div>

          {/* Ödül ve Rating */}
          <div className="w-full py-2 flex justify-between items-center">
            <div className="text-2xl font-bold text-white mb-2">{task.amount}</div>
            <div className="flex flex-col items-center justify-center">
              <span className="text-gray-400 text-sm">Popülerlik Puanı</span>
              {/* Yıldızlar */}
              <div className="flex items-center justify-center">
                {[1, 2, 3, 4, 5].map((star) => (
                  star <= (task.rating || 4)
                    ? <Icon key={star} name="star_ico" size={20} />
                    : <Iconify key={star} icon="mdi:star" size={20} className="text-gray-600" />
                ))}
              </div>
            </div>
          </div>

          {/* CTA Button */}
          <Button variant="primary" size="lg" fullWidth className="mb-4">
            Görevi Yap ve {task.amount} Kazan
          </Button>
        </div>

        {/* Sekmeler */}
        <div className="flex gap-6 border-b border-base-3 mb-2">
          <button
            className={`text-white text-base font-medium pb-2 border-b-2 transition-all ${
              activeTab === 'oduller' ? 'border-primary' : 'border-transparent text-white/40'
            }`}
            onClick={() => setActiveTab('oduller')}
          >
            Ödüller
          </button>
          <button
            className={`text-white text-base font-medium pb-2 border-b-2 transition-all ${
              activeTab === 'detay' ? 'border-primary' : 'border-transparent text-white/40'
            }`}
            onClick={() => setActiveTab('detay')}
          >
            Detay
          </button>
        </div>

        {/* Sekme İçeriği */}
        {activeTab === 'oduller' ? (
          <div className="space-y-2 mt-2">
            {(task.rewards || [
              { amount: '0,15₺', label: 'Complete Event 1', completed: false },
              { amount: '0,15₺', label: 'Complete Event 2', completed: false },
              { amount: '0,15₺', label: 'Complete Event 3', completed: true },
              { amount: '0,15₺', label: 'Complete Event 4', completed: true },
              { amount: '0,15₺', label: 'Complete Event 5', completed: true },
            ] as TaskReward[]).map((reward: TaskReward, idx: number) => (
              <div key={idx} className="flex items-center justify-between bg-base-2 rounded-xl px-2 py-2">
                <div className="flex items-center gap-2">
                  <span className="bg-base-3 text-white text-xs font-semibold px-4 py-1 rounded-lg">
                    {reward.amount}
                  </span>
                  <span className="text-white text-sm">{reward.label}</span>
                </div>
                {reward.completed ? (
                  ' '
                ) : (
                  <Icon name="mdi:marker-tick" size={22} className="text-gray-600" />
                )}
              </div>
            ))}
          </div>
        ) : (
          <div className="mt-2 space-y-4">
            <div className="bg-base-2 p-2 rounded-2xl flex flex-col space-y-5">
              {/* Görev esnekliği */}
              {task.taskFlexibility && (
                <div className="flex items-start gap-3">
                  <div className="w-10 h-10 rounded-xl bg-white/5 flex items-center justify-center flex-shrink-0">
                    <Iconify icon="mdi:refresh" size={20} className="text-white" />
                  </div>
                  <div className="flex-1">
                    <h3 className="text-white font-semibold text-sm mb-1">
                      {task.taskFlexibility.title}
                    </h3>
                    <p className="text-gray-400 text-sm">
                      {task.taskFlexibility.description}
                    </p>
                  </div>
                </div>
              )}

              {/* Sadece Yeni Üyeler */}
              {task.newMembers && (
                <div className="flex items-start gap-3">
                  <div className="w-10 h-10 rounded-xl bg-white/5 flex items-center justify-center flex-shrink-0">
                    <Iconify icon="mdi:account-multiple-plus" size={20} className="text-white" />
                  </div>
                  <div className="flex-1">
                    <h3 className="text-white font-semibold text-sm mb-1">
                      {task.newMembers.title}
                    </h3>
                    <p className="text-gray-400 text-sm">
                      {task.newMembers.description}
                    </p>
                  </div>
                </div>
              )}
            </div>

            {/* Açıklama */}
            {task.description && (
              <div className="flex items-start gap-3 bg-base-2 p-2 rounded-2xl">
                <div className="w-10 h-10 bg-white/5 rounded-xl flex items-center justify-center flex-shrink-0">
                  <Iconify icon="mdi:text-box-outline" size={20} className="text-white" />
                </div>
                <div className="flex-1">
                  <h3 className="text-white font-semibold text-sm mb-1">
                    {task.description.title}
                  </h3>
                  <p className="text-gray-400 text-sm">
                    {task.description.description}
                  </p>
                </div>
              </div>
            )}
          </div>
        )}
      </motion.div>
    </Modal>
  )
}
