'use client'

import React, { useState } from 'react'
import { useRouter } from 'next/navigation'
import { motion } from 'framer-motion'
import { Modal, Button, Input, Iconify, Checkbox } from '../ui'

interface LoginModalProps {
  isOpen: boolean
  onClose: () => void
  onSwitchToRegister: () => void
  onForgotPassword: () => void
}

export function LoginModal({ isOpen, onClose, onSwitchToRegister, onForgotPassword }: LoginModalProps) {
  const router = useRouter()
  const [formData, setFormData] = useState({
    email: '',
    password: '',
    rememberMe: false
  })
  const [isLoading, setIsLoading] = useState(false)
  const [errors, setErrors] = useState<Record<string, string>>({})

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type, checked } = e.target
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? checked : value
    }))
    
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }))
    }
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setErrors({})

    // Basic validation
    const newErrors: Record<string, string> = {}
    
    if (!formData.email) {
      newErrors.email = 'E-posta adresi gerekli'
    } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
      newErrors.email = 'Geçerli bir e-posta adresi girin'
    }
    
    if (!formData.password) {
      newErrors.password = 'Şifre gerekli'
    } else if (formData.password.length < 6) {
      newErrors.password = 'Şifre en az 6 karakter olmalı'
    }

    if (Object.keys(newErrors).length > 0) {
      setErrors(newErrors)
      setIsLoading(false)
      return
    }

    try {
      // Simulate API call
      console.log('Login attempt:', formData)
      await new Promise(resolve => setTimeout(resolve, 1000))
      
      // Success - close modal and redirect to home
      onClose()
      router.push('/home')
      
    } catch (error) {
      setErrors({ general: 'Giriş yapılırken bir hata oluştu' })
    } finally {
      setIsLoading(false)
    }
  }

  const handleSocialLogin = (provider: 'google' | 'apple' | 'facebook') => {
    console.log(`${provider} ile giriş yapılıyor...`)
    // Handle social login
  }

  return (
    <Modal 
      isOpen={isOpen} 
      onClose={onClose}
      size="md"
      image='/assets/images/login-banner.png'
      mobileImage='/assets/images/login-banner-mobile.png'
      fullscreen
    >
      <motion.div 
        className="space-y-4 md:space-y-6"
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3 }}
      >
        <motion.h2 
          className="text-xl md:text-2xl font-semibold text-white text-start"
          initial={{ opacity: 0, y: -10 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.1, duration: 0.3 }}
        >
          Giriş Yap
        </motion.h2>
        <form onSubmit={handleSubmit} className="space-y-3 md:space-y-4">
        {errors.general && (
          <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
            <p className="text-sm text-red-400">{errors.general}</p>
          </div>
        )}

        <Input
          name="email"
          type="email"
          label="E-posta Adresi"
          placeholder="ornek@email.com"
          value={formData.email}
          onChange={handleInputChange}
          error={errors.email}
          required
          //leftIcon={<Iconify icon="material-symbols:mail-outline" size={20} />}
        />

        <Input
          name="password"
          type="password"
          label="Şifre"
          placeholder="Şifrenizi girin"
          value={formData.password}
          onChange={handleInputChange}
          error={errors.password}
          required
          showPasswordToggle
          //leftIcon={<Iconify icon="material-symbols:lock-outline" size={20} />}
        />

        <div className="flex items-center justify-between">
          <Checkbox
            name="rememberMe"
            checked={formData.rememberMe}
            onChange={(checked) => setFormData(prev => ({ ...prev, rememberMe: checked }))}
            label="Beni hatırla"
            size="md"
            variant="primary"
          />
          
          <button
            type="button"
            onClick={onForgotPassword}
            className="text-sm text-primary hover:text-primary/80 transition-colors"
          >
            Şifremi unuttum
          </button>
        </div>

        <Button
          type="submit"
          variant="primary"
          size="md"
          className="w-full"
          disabled={isLoading}
        >
          {isLoading ? (
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
              <span className="text-sm md:text-base">Giriş yapılıyor...</span>
            </div>
          ) : (
            <span className="text-sm md:text-base">Giriş Yap</span>
          )}
        </Button>

        <div className="relative my-4 md:my-6">
          <div className="absolute inset-0 flex items-center">
            <div className="w-full border-t border-base-3" />
          </div>
          <div className="relative flex justify-center text-xs md:text-sm">
            <span className="px-3 md:px-4 bg-base-1 text-gray-400">veya</span>
          </div>
        </div>

        <div className="grid grid-cols-2 gap-2 md:gap-3">
          {/* Google - Tam genişlik */}
          <Button
            type="button"
            variant="outline"
            onClick={() => handleSocialLogin('google')}
            className="text-xs md:text-sm col-span-2"
            fullWidth
          >
            <Iconify icon="logos:google-icon" size={16} className="md:text-[18px]" />
            <span className="hidden sm:inline">Google</span>
          </Button>
          
          {/* Facebook - Yarı genişlik */}
          <Button
            type="button"
            variant="outline"
            onClick={() => handleSocialLogin('facebook')}
            className="text-xs md:text-sm"
            fullWidth
          >
            <Iconify icon="logos:facebook" size={16} className="md:text-[18px]" />
            <span className="hidden sm:inline">Facebook</span>
          </Button>

          {/* Apple - Yarı genişlik */}
          <Button
            type="button"
            variant="outline"
            onClick={() => handleSocialLogin('apple')}
            className="text-xs md:text-sm"
            fullWidth
          >
            <Iconify icon="ic:baseline-apple" size={16} className="md:text-[18px]" />
            <span className="hidden sm:inline">Apple</span>
          </Button>
        </div>

        <div className="text-center pt-3 md:pt-4 border-t border-base-3">
          <p className="text-xs md:text-sm text-gray-400">
            Hesabınız yok mu?{' '}
            <button
              type="button"
              onClick={onSwitchToRegister}
              className="text-primary hover:text-primary/80 font-medium transition-colors"
            >
              Kayıt ol
            </button>
          </p>
        </div>
      </form>
      </motion.div>
    </Modal>
  )
}