'use client'

import React, { useState } from 'react'
import { useRouter } from 'next/navigation'
import { motion } from 'framer-motion'
import { Modal, Button, Input } from '../ui'

interface AdditionalInfoModalProps {
  isOpen: boolean
  onClose: () => void
  initialData?: {
    firstName: string
    lastName: string
    password: string
  }
}

export function AdditionalInfoModal({ isOpen, onClose, initialData }: AdditionalInfoModalProps) {
  const router = useRouter()
  const [formData, setFormData] = useState({
    username: '',
    email: '',
    phone: '',
    birthMonth: '',
    birthDay: '',
    birthYear: '',
    gender: '',
    city: ''
  })
  const [isLoading, setIsLoading] = useState(false)
  const [errors, setErrors] = useState<Record<string, string>>({})

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target
    
    setFormData(prev => ({
      ...prev,
      [name]: value
    }))
    
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }))
    }
  }

  const validateForm = () => {
    const newErrors: Record<string, string> = {}
    
    if (!formData.username.trim()) {
      newErrors.username = 'Kullanıcı adı gerekli'
    } else if (formData.username.length < 3) {
      newErrors.username = 'Kullanıcı adı en az 3 karakter olmalı'
    }
    
    if (!formData.email) {
      newErrors.email = 'E-posta adresi gerekli'
    } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
      newErrors.email = 'Geçerli bir e-posta adresi girin'
    }
    
    if (!formData.phone) {
      newErrors.phone = 'Telefon numarası gerekli'
    } else if (!/^\+?[\d\s\-\(\)]{10,}$/.test(formData.phone)) {
      newErrors.phone = 'Geçerli bir telefon numarası girin'
    }
    
    if (!formData.birthMonth || !formData.birthDay || !formData.birthYear) {
      newErrors.birthDate = 'Doğum tarihi gerekli'
    }
    
    if (!formData.gender) {
      newErrors.gender = 'Cinsiyet seçimi gerekli'
    }
    
    if (!formData.city) {
      newErrors.city = 'Şehir gerekli'
    }

    return newErrors
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setErrors({})

    const stepErrors = validateForm()
    if (Object.keys(stepErrors).length > 0) {
      setErrors(stepErrors)
      setIsLoading(false)
      return
    }

    try {
      // Simulate API call
      console.log('Kayıt tamamlanıyor:', { ...initialData, ...formData })
      await new Promise(resolve => setTimeout(resolve, 1500))
      
      // Success - redirect to home
      onClose()
      router.push('/home')
      
    } catch (error) {
      setErrors({ general: 'Kayıt olurken bir hata oluştu' })
    } finally {
      setIsLoading(false)
    }
  }

  // Generate year options (18-80 years old)
  const currentYear = new Date().getFullYear()
  const yearOptions = Array.from({ length: 63 }, (_, i) => currentYear - 18 - i)
  
  // Generate day options
  const dayOptions = Array.from({ length: 31 }, (_, i) => i + 1)
  
  // Month options
  const monthOptions = [
    { value: '1', label: 'Ocak' },
    { value: '2', label: 'Şubat' },
    { value: '3', label: 'Mart' },
    { value: '4', label: 'Nisan' },
    { value: '5', label: 'Mayıs' },
    { value: '6', label: 'Haziran' },
    { value: '7', label: 'Temmuz' },
    { value: '8', label: 'Ağustos' },
    { value: '9', label: 'Eylül' },
    { value: '10', label: 'Ekim' },
    { value: '11', label: 'Kasım' },
    { value: '12', label: 'Aralık' }
  ]

  // Türkiye'deki 81 il
  const cityOptions = [
    'Adana', 'Adıyaman', 'Afyonkarahisar', 'Ağrı', 'Amasya', 'Ankara', 'Antalya', 'Artvin',
    'Aydın', 'Balıkesir', 'Bilecik', 'Bingöl', 'Bitlis', 'Bolu', 'Burdur', 'Bursa',
    'Çanakkale', 'Çankırı', 'Çorum', 'Denizli', 'Diyarbakır', 'Edirne', 'Elazığ', 'Erzincan',
    'Erzurum', 'Eskişehir', 'Gaziantep', 'Giresun', 'Gümüşhane', 'Hakkari', 'Hatay', 'Isparta',
    'Mersin', 'İstanbul', 'İzmir', 'Kars', 'Kastamonu', 'Kayseri', 'Kırklareli', 'Kırşehir',
    'Kocaeli', 'Konya', 'Kütahya', 'Malatya', 'Manisa', 'Kahramanmaraş', 'Mardin', 'Muğla',
    'Muş', 'Nevşehir', 'Niğde', 'Ordu', 'Rize', 'Sakarya', 'Samsun', 'Siirt',
    'Sinop', 'Sivas', 'Tekirdağ', 'Tokat', 'Trabzon', 'Tunceli', 'Şanlıurfa', 'Uşak',
    'Van', 'Yozgat', 'Zonguldak', 'Aksaray', 'Bayburt', 'Karaman', 'Kırıkkale', 'Batman',
    'Şırnak', 'Bartın', 'Ardahan', 'Iğdır', 'Yalova', 'Karabük', 'Kilis', 'Osmaniye', 'Düzce'
  ]

  return (
    <Modal 
      isOpen={isOpen} 
      onClose={onClose}
      size="md"
    >
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3 }}
        className="space-y-4"
      >
        <h2 className="text-2xl font-semibold text-white text-start w-full pr-8">
          Daha Fazla Bilgi
        </h2>
        
        <form onSubmit={handleSubmit} className="space-y-4">
          {errors.general && (
            <div className="p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
              <p className="text-sm text-red-400">{errors.general}</p>
            </div>
          )}

          <Input
            name="username"
            type="text"
            label="Kullanıcı Adı"
            placeholder="Benzersiz kullanıcı adınız"
            value={formData.username}
            onChange={handleInputChange}
            error={errors.username}
            required
            hint="En az 3 karakter, sadece harf, rakam ve alt çizgi"
          />

          <Input
            name="email"
            type="email"
            label="E-posta Adresi"
            placeholder="ornek@email.com"
            value={formData.email}
            onChange={handleInputChange}
            error={errors.email}
            required
          />

          <Input
            name="phone"
            type="tel"
            label="Telefon Numarası"
            placeholder="+90 5xx xxx xx xx"
            value={formData.phone}
            onChange={handleInputChange}
            error={errors.phone}
            required
          />

          {/* Birth Date */}
          <div className="space-y-2">
            <label className="block text-sm font-medium text-gray-200">
              Doğum Tarihi <span className="text-red-400">*</span>
            </label>
            <div className="grid grid-cols-3 gap-3">
              <select
                name="birthDay"
                value={formData.birthDay}
                onChange={handleInputChange}
                className="px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
                required
              >
                <option value="">Gün</option>
                {dayOptions.map(day => (
                  <option key={day} value={day}>{day}</option>
                ))}
              </select>
              
              <select
                name="birthMonth"
                value={formData.birthMonth}
                onChange={handleInputChange}
                className="px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
                required
              >
                <option value="">Ay</option>
                {monthOptions.map(month => (
                  <option key={month.value} value={month.value}>{month.label}</option>
                ))}
              </select>
              
              <select
                name="birthYear"
                value={formData.birthYear}
                onChange={handleInputChange}
                className="px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
                required
              >
                <option value="">Yıl</option>
                {yearOptions.map(year => (
                  <option key={year} value={year}>{year}</option>
                ))}
              </select>
            </div>
            {errors.birthDate && (
              <p className="text-sm text-red-400">{errors.birthDate}</p>
            )}
          </div>

          {/* Gender */}
          <div className="space-y-2">
            <label className="block text-sm font-medium text-gray-200">
              Cinsiyet <span className="text-red-400">*</span>
            </label>
            <select
              name="gender"
              value={formData.gender}
              onChange={handleInputChange}
              className="w-full px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
              required
            >
              <option value="">Cinsiyet seçin</option>
              <option value="male">Erkek</option>
              <option value="female">Kadın</option>
            </select>
            {errors.gender && (
              <p className="text-sm text-red-400">{errors.gender}</p>
            )}
          </div>

          {/* Şehir */}
          <div className="space-y-2">
            <label className="block text-sm font-medium text-gray-200">
              Şehir <span className="text-red-400">*</span>
            </label>
            <select
              name="city"
              value={formData.city}
              onChange={handleInputChange}
              className="w-full px-4 py-3 rounded-xl border bg-base-2 border-base-3 text-white focus:border-primary focus:ring-2 focus:ring-primary/20 focus:outline-none"
              required
            >
              <option value="">Şehir seçin</option>
              {cityOptions.map(city => (
                <option key={city} value={city}>{city}</option>
              ))}
            </select>
            {errors.city && (
              <p className="text-sm text-red-400">{errors.city}</p>
            )}
          </div>

          <Button
            type="submit"
            variant="primary"
            size="md"
            fullWidth
            disabled={isLoading}
          >
            {isLoading ? (
              <div className="flex items-center gap-2">
                <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                Kayıt Tamamlanıyor...
              </div>
            ) : (
              'Kayıt Tamamla'
            )}
          </Button>
        </form>
      </motion.div>
    </Modal>
  )
}
