'use client'

import React from 'react'
import { cn } from '@/lib/utils'
import { Icon, Iconify } from '../ui'
import type { IconName } from '@/lib/icons'
import Image from 'next/image'

interface SidebarProps {
  className?: string
  isMobileOpen?: boolean
  onClose?: () => void
  onOpenMobileAppModal?: () => void
}

interface MenuItem {
  title: string
  href: string
  icon?: IconName
  badge?: string
  hideOnMobile?: boolean
}

const Sidebar: React.FC<SidebarProps> = ({ className, isMobileOpen = false, onClose, onOpenMobileAppModal }) => {
  const mainMenuItems: MenuItem[] = [
  { title: 'Ana Sayfa', href: '/home', icon:'fa6-solid:list', hideOnMobile: true },
  { title: 'Makro Görevler', href: '/macro-tasks',icon:'task-1' },
  { title: 'Mikro Görevler', href: '/micro-tasks',icon:'task-2' },
  { title: 'Dinle Kazan', href: '#', icon:'fluent:headphones-sound-wave-24-filled' },
  { title: 'Görevlerim', href: '/my-duties',icon:'heroicons:newspaper', hideOnMobile: true },
  { title: 'Bakiye Çek', href: '/wallet',icon:'tdesign:money-filled', hideOnMobile: true },
  { title: 'Liderlik Sıralaması', href: '/leaderboard',icon:'material-symbols:leaderboard-rounded' },
  { title: 'Referanslarım', href: '/references',icon:'mdi:user-plus' },
  { title: 'Referans Ödülleri', href: '/references',icon:'flowbite:gift-box-outline' },
]

const bottomMenuItems: MenuItem[] = [
  { title: 'Destek Taleplerim', href: '/support',icon:'streamline-plump:customer-support-3-solid' },
  { title: 'Blog', href: '/blog',icon:'mdi:blog-outline' },
  { title: 'Sıkça Sorulan Sorula', href: '/faq',icon:'wpf:faq' },
  { title: 'Ayarlar', href: '/settings',icon:'weui:setting-filled' },
]


  return (
    <>
      {/* Mobile Overlay */}
      {isMobileOpen && (
        <div 
          className="fixed inset-0 bg-black/60 backdrop-blur-sm z-40 lg:hidden"
          onClick={onClose}
        />
      )}

      {/* Sidebar */}
      <aside className={cn(
        'pt-10 md:pt-15  h-screen w-64 bg-base-1 border-r border-primary/20',
        'overflow-y-auto scrollbar-hide scrollbar-transparent',
        'fixed lg:sticky top-0  ',
        'transition-transform duration-300 ease-in-out',
        
        // Mobile: slide from right
        'lg:translate-x-0 ',
        isMobileOpen ? 'translate-x-0 z-50' : 'translate-x-full lg:translate-x-0',
        // Position on mobile
        'right-0 lg:right-auto',
        className
      )}>
        {/* Mobile Close Button */}
        <button
          onClick={onClose}
          className="lg:hidden absolute   top-2 right-2 w-10 h-10 flex items-center justify-center text-white hover:bg-base-2 rounded-lg transition-colors"
        >
          <Iconify className='' icon="mdi:close" size={24} />
        </button>

        <div className="flex flex-col h-full">
        {/* Header Card - Dinle Kazan */}
        <div className="p-3">
        <Image
          src="/assets/images/sidebar_banner.png" // public klasöründen erişim
          alt="Profil Resmi"
          width={300}
          height={300}
          unoptimized
          quality={100}
          className="w-full h-auto rounded-xl"
        />
        </div>

        {/* Main Navigation */}
        <nav className="flex-1 px-3">
          <ul className="space-y-0.5">
            {mainMenuItems
              .filter(item => !(item.hideOnMobile && isMobileOpen))
              .map((item, index) => (
              <li key={index}>
                <a
                  href={item.href}
                  onClick={(e) => {
                    if (item.title === 'Dinle Kazan') {
                      e.preventDefault()
                      onOpenMobileAppModal?.()
                    }
                  }}
                  className={cn(
                    'flex items-center gap-3 p-1.5 rounded-xl text-sm font-medium transition-all duration-200',
                    'text-gray-300 hover:text-white hover:bg-base-2',
                    'group relative'
                  )}
                >
                  {/* Icon */}
                  <div className="w-8 h-8 rounded-md group-hover:bg-base-3 flex items-center justify-center flex-shrink-0 transition-all duration-200">
                    {item.icon && <Icon name={item.icon} size="sm" className="text-gray-300 group-hover:text-white" />}
                  </div>

                  <span className="flex-1 text-white/50 group-hover:text-white/80 text-sm line-clamp-1">{item.title}</span>

                  {item.badge && (
                    <span className="px-2 py-1 bg-primary text-xs font-semibold rounded-full">
                      {item.badge}
                    </span>
                  )}
                </a>
              </li>
            ))}
          </ul>
        </nav>

        {/* Bottom Navigation */}
        <div className="p-3 border-t border-base-3">
          <ul className="space-y-1 pb-28 md:pb-0">
            {bottomMenuItems.map((item, index) => (
              <li key={index}>
                <a
                  href={item.href}
                  className={cn(
                    'flex items-center gap-3 p-1.5 rounded-xl text-sm font-medium transition-all duration-200',
                    'text-gray-400 hover:text-white hover:bg-base-2',
                    'group'
                  )}
                >
                  {/* Icon */}
                  <div className="w-8 h-8 rounded-md group-hover:bg-base-3 flex items-center justify-center flex-shrink-0 transition-all duration-200">
                    {item.icon && <Icon name={item.icon} size="md" className="text-gray-400 group-hover:text-white" />}
                  </div>
                  <span className="flex-1 text-sm line-clamp-1">{item.title}</span>
                </a>
              </li>
            ))}
          </ul>
        </div>
      </div>
    </aside>
    </>
  )
}

export default Sidebar