'use client'

import React from 'react'
import Image from 'next/image'
import { cn } from '@/lib/utils'
import { Iconify } from '../ui'

interface FooterProps {
  className?: string
}

interface FooterSection {
  title: string
  links: {
    label: string
    href: string
  }[]
}

interface SocialLink {
  name: string
  icon: string
  href: string
}

const Footer: React.FC<FooterProps> = ({ className }) => {
  const footerSections: FooterSection[] = [
    {
      title: 'Hakkında',
      links: [
        { label: 'Kiqro Hakkında', href: '/about' },
      ]
    },
    {
      title: 'Güvenlik',
      links: [
        { label: 'Şartları Özeti', href: '/terms-summary' },
        { label: 'Güvenlik Koşulları', href: '/security-terms' },
        { label: 'Spotify Kullanımı', href: '/spotify-usage' },
        { label: 'Hesap Silme', href: '/delete-account' },
      ]
    },
    {
      title: 'Gizlilik',
      links: [
        { label: 'Genel Aydınlatma Metni', href: '/privacy-policy' },
        { label: 'Üyelik Sözleşmesi', href: '/membership-agreement' },
        { label: 'Site Kullanım Şartları', href: '/terms-of-use' },
        { label: 'Çerez Politikası', href: '/cookie-policy' },
      ]
    },
    {
      title: 'Destek',
      links: [
        { label: 'İletişim', href: '/contact' },
        { label: 'S.S.S', href: '/faq' },
      ]
    }
  ]

  const socialLinks: SocialLink[] = [
    { name: 'Instagram', icon: 'mdi:instagram', href: 'https://instagram.com' },
    { name: 'Twitter', icon: 'mdi:twitter', href: 'https://twitter.com' },
    { name: 'TikTok', icon: 'ic:baseline-tiktok', href: 'https://tiktok.com' },
    { name: 'LinkedIn', icon: 'mdi:linkedin', href: 'https://linkedin.com' },
    { name: 'Facebook', icon: 'mdi:facebook', href: 'https://facebook.com' },
  ]

  return (
    <footer className={cn(
      'bg-base-1 mx-4 mt-10 rounded-b-0 rounded-t-2xl border border-primary/10',
      className
    )}>
      <div className="mx-auto px-4 sm:px-6 lg:px-16 py-8 md:py-12">
        {/* Main Footer Content */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 md:gap-8">
          {/* Logo Section */}
          <div className="lg:col-span-1">
            <div className="flex flex-col gap-4">
              <div className="h-9 flex items-center">
                <Image
                  src="/assets/logo.png"
                  alt="Kiqro Logo"
                  width={120}
                  height={36}
                  unoptimized
                  className="h-8 md:h-9 w-auto object-contain"
                />
              </div>
              
              {/* Social Links */}
              <div className="flex items-center gap-2">
                {socialLinks.map((social, index) => (
                  <a
                    key={index}
                    href={social.href}
                    target="_blank"
                    rel="noopener noreferrer"
                    className={cn(
                      'w-9 h-9 rounded-full flex items-center justify-center',
                      'bg-primary/10 hover:bg-primary/20',
                      'transition-all duration-200'
                    )}
                  >
                    <Iconify 
                      icon={social.icon}
                      size={20}
                      className="text-gray-400 hover:text-white transition-colors duration-200"
                    />
                  </a>
                ))}
              </div>
            </div>
          </div>

          {/* Footer Sections */}
          {footerSections.map((section, index) => (
            <div key={index} className="lg:col-span-1">
              <h2 className="text-white font-semibold text-base md:text-lg mb-3 md:mb-4">
                {section.title}
              </h2>
              <ul className="space-y-2 md:space-y-3">
                {section.links.map((link, linkIndex) => (
                  <li key={linkIndex}>
                    <a
                      href={link.href}
                      className={cn(
                        'text-gray-400 hover:text-white text-sm',
                        'transition-colors duration-200',
                        'block hover:underline'
                      )}
                    >
                      {link.label}
                    </a>
                  </li>
                ))}
              </ul>
            </div>
          ))}
        </div>

        {/* Bottom Section */}
        <div className="mt-8 md:mt-12 pt-6 md:pt-8 border-t border-base-2">
          <div className="flex flex-col md:flex-row justify-between items-center gap-3 md:gap-4">
            {/* Left side - Legal Links */}
            <div className="flex flex-col md:flex-row items-center gap-2 md:gap-6 text-xs md:text-sm text-center md:text-left">
              <a
                href="/privacy-policy"
                className="text-gray-400 hover:text-white transition-colors duration-200"
              >
                Gizlilik Politikası ve Aydınlatma Metni
              </a>
              <span className="hidden md:inline text-gray-600">|</span>
              <a
                href="/cookie-policy"
                className="text-gray-400 hover:text-white transition-colors duration-200"
              >
                Çerez Politikası
              </a>
            </div>

            {/* Right side - Copyright */}
            <div className="text-xs md:text-sm text-gray-400 text-center md:text-right">
              Her hakkı saklıdır. ©2025 kiqro
            </div>
          </div>
        </div>
      </div>
    </footer>
  )
}

export default Footer