'use client'

import React, { useState, useEffect } from 'react'
import { Icon } from '@/components/ui'
import Image from 'next/image'
import { getInitialPayments, generateLivePayment, getTotalWithdrawalsToday, type LivePayment } from '@/data'

interface LivePaymentSectionProps {
  imageSrc?: string
  imageAlt?: string
  title?: string
  className?: string
  autoUpdate?: boolean
  updateInterval?: number
}

export function LivePaymentSection({
  imageSrc = './assets/images/bu_ay_ödenen.png',
  imageAlt = 'Bu Ay Ödenen',
  title = 'Canlı Para Çekimi',
  className = '',
  autoUpdate = true,
  updateInterval = 10000
}: LivePaymentSectionProps) {
  const [payments, setPayments] = useState<LivePayment[]>([])
  const [totalWithdrawals, setTotalWithdrawals] = useState(0)
  const [newPaymentId, setNewPaymentId] = useState<string | null>(null)

  useEffect(() => {
    setPayments(getInitialPayments())
    setTotalWithdrawals(getTotalWithdrawalsToday())
  }, [])

  useEffect(() => {
    if (!autoUpdate) return

    const interval = setInterval(() => {
      const newPayment = generateLivePayment()
      
      setPayments(prev => {
        const updated = [newPayment, ...prev]
        return updated.slice(0, 4)
      })
      
      setNewPaymentId(newPayment.id)
      
      setTimeout(() => {
        setNewPaymentId(null)
      }, 500)
      
      setTotalWithdrawals(getTotalWithdrawalsToday())
    }, updateInterval)

    return () => clearInterval(interval)
  }, [autoUpdate, updateInterval])

  return (
    <section className={`w-full py-8 md:-mt-15 px-4 ${className}`}>
      <div className="flex flex-col justify-center items-center w-full">
        <div className="mt-6 border md:mt-10 w-full max-w-6xl p-4 md:p-6 rounded-2xl md:rounded-3xl bg-base-2 border-base-3">
          
          <div className="flex flex-col space-y-6 md:hidden">
            <div className="w-full h-[200px] relative rounded-xl overflow-hidden">
              <Image
                src={imageSrc}
                alt={imageAlt}
                fill
                className="object-cover"
                quality={100}
                unoptimized
              />
            </div>

            <div className="flex flex-col">
              <div className="flex items-center justify-center space-x-2 mb-4">
                <span className="w-2 h-2 rounded-full bg-green-400 animate-pulse" />
                <h2 className="text-base font-semibold">{title}</h2>
              </div>

              <div className="space-y-2 relative">
                {payments.map((payment, index) => (
                  <div
                    key={payment.id}
                    className={`bg-white/5 p-2 rounded-xl flex justify-between items-center hover:bg-white/10 transition-all duration-500 ease-in-out ${
                      newPaymentId === payment.id 
                        ? 'animate-in slide-in-from-top-2 fade-in' 
                        : ''
                    }`}
                  >
                    <div className="flex items-center space-x-2">
                      <div className="relative w-8 h-8 flex-shrink-0">
                        <Image
                          src={payment.avatar}
                          alt={payment.name}
                          fill
                          className="rounded-full border border-primary/20 object-cover"
                        />
                      </div>
                      <div className="flex flex-col">
                        <span className="text-xs font-medium">{payment.name}</span>
                        <span className="text-[10px] text-white/50">{payment.timeAgo}</span>
                      </div>
                    </div>
                    <div className="flex items-center space-x-1">
                      <Icon name={payment.paymentIcon} size="sm" />
                      <span className="text-xs font-semibold">{payment.amount}</span>
                    </div>
                  </div>
                ))}
              </div>

              {payments.length > 0 && (
                <div className="mt-4 text-center">
                  <p className="text-[10px] text-white/40">
                    Son 24 saatte {totalWithdrawals} kişi daha para çekti
                  </p>
                </div>
              )}
            </div>
          </div>

          <div className="hidden md:flex md:space-x-6 px-6 lg:space-x-10 min-h-[280px]">
            <div className="flex-shrink-0 w-[340px] lg:w-[380px] relative">
              <Image
                src={imageSrc}
                alt={imageAlt}
                fill
                className="object-contain rounded-2xl"
                quality={100}
                unoptimized
              />
            </div>

            <div className="flex-1 flex flex-col">
              <div className="flex items-center space-x-2 mb-4">
                <span className="w-2 h-2 rounded-full bg-green-400 animate-pulse" />
                <h2 className="text-lg font-semibold">{title}</h2>
              </div>

              <div className="space-y-2 flex-1 relative">
                {payments.map((payment, index) => (
                  <div
                    key={payment.id}
                    className={`bg-white/5 p-3 rounded-xl flex justify-between items-center hover:bg-white/10 transition-all duration-500 ease-in-out ${
                      newPaymentId === payment.id 
                        ? 'animate-in slide-in-from-top-4 fade-in scale-in-95' 
                        : ''
                    }`}
                  >
                    <div className="flex items-center space-x-3">
                      <div className="relative w-10 h-10 flex-shrink-0">
                        <Image
                          src={payment.avatar}
                          alt={payment.name}
                          fill
                          className="rounded-full border border-primary/20 object-cover"
                        />
                      </div>
                      <div className="flex flex-col">
                        <span className="text-sm font-medium">{payment.name}</span>
                        <span className="text-xs text-white/50">{payment.timeAgo}</span>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Icon name={payment.paymentIcon} size="sm" className="text-base" />
                      <span className="opacity-20">|</span>
                      <span className="text-sm font-semibold">{payment.amount}</span>
                    </div>
                  </div>
                ))}
              </div>

              {payments.length > 0 && (
                <div className="mt-4 text-center">
                  <p className="text-xs text-white/40">
                    Son 24 saatte {totalWithdrawals} kişi daha para çekti
                  </p>
                </div>
              )}
            </div>
          </div>

        </div>
      </div>
    </section>
  )
}
