'use client'

import React from 'react'
import { AppCard, type AppCardSize } from './AppCard'

export interface AppCardData {
  /** Uygulama adı */
  appName: string
  /** Logo URL'i */
  logo: string
  /** Görev açıklaması */
  taskDescription: string
  /** Ödül miktarı */
  rewardAmount: string
  /** Rating puanı */
  rating: number
  /** Kart boyutu */
  size: AppCardSize
}

export interface HeroSectionProps {
  /** Ana başlık */
  mainTitle?: string
  /** Vurgulanan başlık kısmı */
  highlightedTitle?: string
  /** AppCard verileri */
  appCards?: AppCardData[]
  /** Kart tıklama handler'ı */
  onCardClick?: (appName: string, size: AppCardSize) => void
  /** Container genişliği */
  containerWidth?: number
  /** Ek CSS sınıfları */
  className?: string
}

/**
 * HeroSection Component
 * Header Start - Header End arasındaki ana section
 * Başlık ve AppCard grid'ini içeriyor
 */
const HeroSection: React.FC<HeroSectionProps> = ({
  mainTitle = "görevlerini yaparak ek gelir elde edin.",
  highlightedTitle = "En yeni uygulama, oyun, anket ve müzik",
  appCards = [
    {
      appName: "Spotify",
      logo: "https://cdn.pixabay.com/photo/2021/10/24/21/23/instagram-6739335_1280.jpg",
      taskDescription: "3dk Şarkı Dinle",
      rewardAmount: "$64",
      rating: 5.2,
      size: "sm"
    },
    {
      appName: "Spotify",
      logo: "https://cdn.pixabay.com/photo/2016/10/22/00/15/spotify-1759471_1280.jpg",
      taskDescription: "3dk Şarkı Dinle",
      rewardAmount: "$64",
      rating: 5.2,
      size: "base"
    },
    {
      appName: "Spotify",
      logo: "https://cdn.pixabay.com/photo/2016/10/22/00/15/spotify-1759471_1280.jpg",
      taskDescription: "3dk Şarkı Dinle",
      rewardAmount: "$64",
      rating: 5.2,
      size: "md"
    },
    {
      appName: "Spotify",
      logo: "https://cdn.pixabay.com/photo/2016/10/22/00/15/spotify-1759471_1280.jpg",
      taskDescription: "3dk Şarkı Dinle",
      rewardAmount: "$64",
      rating: 5.2,
      size: "base"
    },
    {
      appName: "Spotify",
      logo: "https://cdn.pixabay.com/photo/2021/10/24/21/23/instagram-6739335_1280.jpg",
      taskDescription: "3dk Şarkı Dinle",
      rewardAmount: "$64",
      rating: 5.2,
      size: "sm"
    }
  ],
  onCardClick,
  containerWidth = 1000,
  className
}) => {
  const handleCardClick = (appName: string, size: AppCardSize) => {
    if (onCardClick) {
      onCardClick(appName, size)
    } else {
      console.log(`${appName} ${size.toUpperCase()} görevine tıklandı!`)
    }
  }

  return (
    <div className={className}>
      {/* Header */}
      <div className="text-center mb-6 md:mb-12 w-full max-w-full md:max-w-[1000px] px-4 mx-auto">
        <h1 className="text-xl sm:text-2xl md:text-3xl lg:text-4xl space-y-1 sm:space-y-1.5 md:space-y-2 flex flex-col font-bold mb-4 md:mb-6 leading-tight">
          <span className="text-[#BDB2FF]">{highlightedTitle}</span>
          <span className="text-white">{mainTitle}</span>
        </h1>
      </div>

      {/* App Cards Grid */}
      {/* Mobile View - 3 cards: left sm, center md, right sm */}
      <div className="flex items-center justify-center gap-2 sm:gap-3 md:hidden w-full px-3 sm:px-4">
        {/* Sol kart - sm */}
        <div className="flex-shrink-0">
          <AppCard
            size="sm"
            appName={appCards[0].appName}
            logo={appCards[0].logo}
            taskDescription={appCards[0].taskDescription}
            rewardAmount={appCards[0].rewardAmount}
            rating={appCards[0].rating}
            onClick={() => handleCardClick(appCards[0].appName, appCards[0].size)}
          />
        </div>
        
        {/* Orta kart - md */}
        <div className="flex-shrink-0">
          <AppCard
            size="md"
            appName={appCards[2].appName}
            logo={appCards[2].logo}
            taskDescription={appCards[2].taskDescription}
            rewardAmount={appCards[2].rewardAmount}
            rating={appCards[2].rating}
            onClick={() => handleCardClick(appCards[2].appName, appCards[2].size)}
          />
        </div>
        
        {/* Sağ kart - sm */}
        <div className="flex-shrink-0">
          <AppCard
            size="sm"
            appName={appCards[4].appName}
            logo={appCards[4].logo}
            taskDescription={appCards[4].taskDescription}
            rewardAmount={appCards[4].rewardAmount}
            rating={appCards[4].rating}
            onClick={() => handleCardClick(appCards[4].appName, appCards[4].size)}
          />
        </div>
      </div>

      {/* Desktop View */}
      <div className="hidden items-center md:grid grid-cols-3 lg:grid-cols-5 gap-4 lg:gap-6 w-full px-6">
        {appCards.map((card, index) => (
          <AppCard
            key={`desktop-${index}`}
            size={card.size}
            appName={card.appName}
            logo={card.logo}
            taskDescription={card.taskDescription}
            rewardAmount={card.rewardAmount}
            rating={card.rating}
            onClick={() => handleCardClick(card.appName, card.size)}
          />
        ))}
      </div>
    </div>
  )
}

export { HeroSection }
export default HeroSection