'use client'

import React from 'react'
import Image from 'next/image'
import { cn } from '@/lib/utils'
import { Iconify } from '@/components/ui'

export type AppCardSize = 'sm' | 'base' | 'md' | 'lg'

export interface AppCardProps {
  /** Uygulama logosu URL'i */
  logo?: string
  /** Uygulama adı */
  appName: string
  /** Görev açıklaması */
  taskDescription: string
  /** Ödül miktarı */
  rewardAmount: string
  /** Uygulama puanı */
  rating?: number
  /** Kart boyutu */
  size?: AppCardSize
  /** Kart tıklama handler'ı */
  onClick?: () => void
  /** Ek CSS sınıfları */
  className?: string
  /** Hover efekti aktif mi */
  hover?: boolean
  /** Tıklanabilir mi */
  clickable?: boolean
}

/**
 * AppCard Component
 * Uygulama görevleri için kullanılan kart bileşeni
 * Spotify, Instagram, TikTok vb. uygulamalar için reusable
 */
const AppCard: React.FC<AppCardProps> = ({
  logo,
  appName,
  taskDescription,
  rewardAmount,
  rating,
  size = 'md',
  onClick,
  className,
  hover = true,
  clickable = true
}) => {
  const handleClick = () => {
    if (clickable && onClick) {
      onClick()
    }
  }

  // Size configurations - responsive
  const sizeConfig = {
    sm: {
      container: 'p-2 sm:p-2.5 md:p-3 space-y-1 sm:space-y-1.5',
      image: 'w-full h-[100px] sm:h-[85px] md:h-[100px]',
      appName: 'text-[11px] sm:text-xs md:text-sm',
      description: 'text-[9px] sm:text-[10px] md:text-xs',
      reward: 'text-xs sm:text-sm md:text-base',
      rating: 'text-[10px] sm:text-[11px] md:text-xs',
      iconSize: 14
    },
    base: {
      container: 'p-2 sm:p-3 md:p-3.5 space-y-1 sm:space-y-1.5',
      image: 'w-full h-[85px] sm:h-[105px] md:h-[125px]',
      appName: 'text-xs sm:text-sm md:text-base',
      description: 'text-[10px] sm:text-xs md:text-sm',
      reward: 'text-sm sm:text-base md:text-lg',
      rating: 'text-[11px] sm:text-xs md:text-sm',
      iconSize: 16
    },
    md: {
      container: 'p-2.5 sm:p-3.5 md:p-4 space-y-1.5 sm:space-y-2',
      image: 'w-full h-[95px] sm:h-[125px] md:h-[150px]',
      appName: 'text-xs sm:text-sm md:text-base',
      description: 'text-[10px] sm:text-xs md:text-sm',
      reward: 'text-base sm:text-lg md:text-xl',
      rating: 'text-xs sm:text-sm md:text-base',
      iconSize: 18
    },
    lg: {
      container: 'p-3 sm:p-4 md:p-6 space-y-2 sm:space-y-2.5 md:space-y-3',
      image: 'w-full h-[110px] sm:h-[160px] md:h-[200px]',
      appName: 'text-sm sm:text-base md:text-lg',
      description: 'text-xs sm:text-sm md:text-base',
      reward: 'text-lg sm:text-xl md:text-2xl',
      rating: 'text-sm sm:text-base md:text-lg',
      iconSize: 20
    }
  }

  const config = sizeConfig[size]

  return (
    <div
      onClick={handleClick}
      className={cn(
        'relative w-full h-fit',
        'bg-white/3 backdrop-blur-sm',
        'rounded-xl md:rounded-2xl transition-all duration-300',
        config.container,
        hover && 'hover:scale-102',
        clickable && 'cursor-pointer',
        className
      )}
    >
      {/* App Logo - 4x4 aspect ratio */}
      <div className="flex justify-center">
        {logo ? (
          <Image
            src={logo}
            alt={`${appName} logo`}
            width={300}
            height={300}
            unoptimized
            quality={100}
            objectFit="contain"
            className={cn(config.image, "rounded-xl md:rounded-2xl object-cover aspect-square")}
          />
        ) : (
          <div className={cn(config.image, "rounded-xl md:rounded-2xl bg-green-500 flex items-center justify-center aspect-square text-xs text-center")}>
            Resim Yüklenemedi
          </div>
        )}
      </div>

      {/* App Name */}
      <h3 className={cn("text-white w-full font-semibold line-clamp-1", config.appName)}>
        {appName}
      </h3>

      {/* Task Description */}
      <p className={cn("text-gray-400 line-clamp-2", config.description)}>
        {taskDescription}
      </p>

      {/* Reward and Rating */}
      <div className="flex justify-between items-center pt-1 sm:pt-1.5 md:pt-2">
        {/* Reward Amount */}
        <div className={cn("text-white font-bold", config.reward)}>
          {rewardAmount}
        </div>

        {/* Rating with Iconify star */}
        {rating && (
          <div className="flex items-center gap-0.5 sm:gap-1">
            <Iconify 
              icon="material-symbols:star-outline" 
              className="text-primary flex-shrink-0"
              size={config.iconSize}
            />
            <span className={cn("text-gray-300 font-medium", config.rating)}>
              {rating}
            </span>
          </div>
        )}
      </div>
    </div>
  )
}

export { AppCard }
export default AppCard