'use client'

import { useMemo } from "react";
import { useRouter } from "next/navigation";
import { Button, Iconify } from "@/components/ui";
import { Footer } from "@/components/layout";
import { isArray } from '@/lib/guards';

interface SupportTicket {
  id: string;
  title: string;
  status: 'reviewing' | 'answered' | 'resolved';
  statusLabel: string;
}

export default function SupportTicketsPage() {
  const router = useRouter();
  
  const tickets: SupportTicket[] = [
    {
      id: '1',
      title: 'Görevim ne zaman onaylanacak',
      status: 'reviewing',
      statusLabel: 'İnceleniyor'
    },
    {
      id: '2',
      title: 'Param gelmedi',
      status: 'answered',
      statusLabel: 'Cevaplandı'
    },
    {
      id: '3',
      title: 'Bakım ne zaman gözükecek',
      status: 'resolved',
      statusLabel: 'Çözümlendi'
    },
    {
      id: '4',
      title: 'Referans ödülüm eksik',
      status: 'reviewing',
      statusLabel: 'İnceleniyor'
    },
    {
      id: '5',
      title: 'Çekim işlemi yapamıyorum',
      status: 'answered',
      statusLabel: 'Cevaplandı'
    },
    {
      id: '6',
      title: 'Hesabım neden kilitlendi?',
      status: 'resolved',
      statusLabel: 'Çözümlendi'
    },
    {
      id: '7',
      title: 'Instagram görevi tamamlanamıyor',
      status: 'reviewing',
      statusLabel: 'İnceleniyor'
    }
  ];

  // Validate tickets data
  const validatedTickets = useMemo(() => {
    return isArray(tickets) && tickets.length > 0 ? tickets : [];
  }, [tickets]);

  const handleCreateTicket = () => {
    router.push('/support/create');
  };

  const handleTicketClick = (ticketId: string) => {
    router.push(`/support/${ticketId}`);
  };

  const getStatusColor = (status: string) => {
    switch(status) {
      case 'reviewing':
        return 'text-blue-400';
      case 'answered':
        return 'text-green-400';
      case 'resolved':
        return 'text-gray-400';
      default:
        return 'text-gray-400';
    }
  };

  return (
    <div className="min-h-screen h-screen overflow-y-auto w-full bg-base-1 pt-20 pb-16">
      <div className="max-w-4xl mx-auto px-4 lg:px-6">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <h1 className="text-white text-2xl lg:text-3xl font-bold">Destek Taleplerim</h1>
          <Button
            variant="primary"
            size="md"
            onClick={handleCreateTicket}
            leftIcon={<Iconify icon="mdi:plus" size={20} />}
          >
            Destek Talebi Oluştur
          </Button>
        </div>

        {/* Tickets List */}
        <div className="space-y-3">
          {validatedTickets.length === 0 ? (
            <div className="text-center py-16 bg-base-2 rounded-2xl">
              <Iconify icon="mdi:ticket-outline" size={64} className="text-gray-600 mx-auto mb-4" />
              <p className="text-gray-400 text-lg mb-2">Henüz destek talebiniz yok</p>
              <p className="text-gray-500 text-sm mb-6">İlk destek talebinizi oluşturarak başlayın</p>
              <Button
                variant="primary"
                size="md"
                onClick={handleCreateTicket}
                leftIcon={<Iconify icon="mdi:plus" size={20} />}
              >
                Destek Talebi Oluştur
              </Button>
            </div>
          ) : (
            <>
              {/* Header Row */}
              <div className="grid grid-cols-[1fr_auto] gap-4 px-4 py-3 text-gray-400 text-sm font-medium">
                <span>Talep Başlığı</span>
                <span>Durum</span>
              </div>

              {/* Ticket Items */}
              {validatedTickets.map((ticket) => (
            <button
              key={ticket.id}
              onClick={() => handleTicketClick(ticket.id)}
              className="w-full bg-base-2 hover:bg-base-3 rounded-2xl p-4 transition-all group"
            >
              <div className="grid grid-cols-[1fr_auto] gap-4 items-center">
                <div className="flex items-center gap-3">
                  <span className="text-white text-base font-medium text-left group-hover:text-primary transition-colors">
                    {ticket.title}
                  </span>
                </div>
                <div className="flex items-center gap-3">
                  <span className={`text-sm font-medium ${getStatusColor(ticket.status)}`}>
                    {ticket.statusLabel}
                  </span>
                  <Iconify 
                    icon="mdi:chevron-right" 
                    size={24} 
                    className="text-gray-400 group-hover:text-primary group-hover:translate-x-1 transition-all"
                  />
                </div>
              </div>
            </button>
          ))}
            </>
          )}
        </div>
      </div>

      <Footer />
    </div>
  );
}
