"use client"

import { useState, useMemo } from "react";
import { Card, CardContent, Iconify, Pagination } from "@/components/ui";
import { Footer } from "@/components/layout";
import { TaskDetailModal } from "@/components/sections";
import { allMacroTasks, MacroTask } from '@/data';
import { usePagination } from '@/lib/hooks';
import type { OfferCardData } from '@/data/offers';
import { isArray } from '@/lib/guards';

export default function MacroTasksPage() {
  // Validate macro tasks data
  const validatedTasks = useMemo(() => {
    return isArray(allMacroTasks) && allMacroTasks.length > 0 
      ? allMacroTasks 
      : [];
  }, []);
  const categories = [
    { id: 'all', label: 'Tümü', icon: 'mdi:view-grid' },
    { id: 'banking', label: 'Bankacılık', icon: 'mdi:bank' },
    { id: 'tech', label: 'Teknoloji', icon: 'mdi:laptop' },
    { id: 'survey', label: 'Anketler', icon: 'mdi:clipboard-list' },
    { id: 'social', label: 'Sosyal Medya', icon: 'mdi:share-variant' },
  ];

  const [selectedCategory, setSelectedCategory] = useState('all');
  const [sortBy, setSortBy] = useState<'highest' | 'lowest' | 'newest'>('highest');

  // Modal state
  const [selectedTask, setSelectedTask] = useState<OfferCardData | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);

    // Filtreleme
  const filteredTasks = useMemo(() => {
    if (!validatedTasks || validatedTasks.length === 0) return [];
    
    let filtered = validatedTasks;
    
    if (selectedCategory !== 'all') {
      filtered = filtered.filter(task => task.category === selectedCategory);
    }

    // Sıralama
    const sorted = [...filtered].sort((a, b) => {
      const amountA = parseFloat(a.amount.replace(/[^0-9.]/g, '')) || 0;
      const amountB = parseFloat(b.amount.replace(/[^0-9.]/g, '')) || 0;
      
      if (sortBy === 'highest') return amountB - amountA;
      if (sortBy === 'lowest') return amountA - amountB;
      return 0; // newest - default order
    });

    return sorted;
  }, [validatedTasks, selectedCategory, sortBy]);

  // Pagination - 21 item per page (3 rows x 7 columns)
  const {
    currentData,
    currentPage,
    totalPages,
    goToPage
  } = usePagination({
    data: filteredTasks,
    itemsPerPage: 21,
    initialPage: 1
  });

  // Kategori veya sıralama değiştiğinde ilk sayfaya dön
  const handleCategoryChange = (categoryId: string) => {
    setSelectedCategory(categoryId);
    goToPage(1); // Reset to first page
  };

  const handleSortChange = (sortValue: 'highest' | 'lowest' | 'newest') => {
    setSortBy(sortValue);
    goToPage(1); // Reset to first page
  };

  // Modal açma fonksiyonu
  const handleCardClick = (task: MacroTask) => {
    // MacroTask'ı OfferCardData formatına dönüştür
    const offerData: OfferCardData = {
      id: task.id.toString(),
      title: task.title,
      subtitle: task.subtitle,
      image: task.image,
      imageAlt: task.title,
      amount: task.amount,
      category: 'green', // Varsayılan kategori
      rating: task.rating || 4,
      rewards: task.rewards || [
        { amount: '0,15₺', label: 'Complete Event 1', completed: false },
        { amount: '0,15₺', label: 'Complete Event 2', completed: false },
        { amount: '0,15₺', label: 'Complete Event 3', completed: true },
        { amount: '0,15₺', label: 'Complete Event 4', completed: true },
        { amount: '0,15₺', label: 'Complete Event 5', completed: true },
      ],
      taskFlexibility: task.taskFlexibility,
      newMembers: task.newMembers,
      description: task.description || {
        title: 'Açıklama',
        description: task.detail || 'Görev detayı burada gösterilecektir. Bu görevi tamamlayarak kazanç elde edebilirsiniz.'
      },
    };
    
    setSelectedTask(offerData);
    setIsModalOpen(true);
  };

  const handleModalClose = () => {
    setIsModalOpen(false);
    setSelectedTask(null);
  };

  return (
    <div className="min-h-screen h-screen overflow-y-auto w-full bg-base-1  pt-18">
      <div className=" p-4 mx-auto">
        {/* Filters */}
        <div className="mb-6 space-y-4 md:space-y-0 md:flex md:items-center md:justify-between md:gap-6">
          {/* Category Filter */}
          <div className="overflow-x-auto scrollbar-hide -mx-4 px-4 md:mx-0 md:px-0 md:flex-1">
            <div className="flex gap-2 md:gap-3 min-w-max md:min-w-0 md:flex-wrap">
              {categories.map((category) => (
                <button
                  key={category.id}
                  onClick={() => handleCategoryChange(category.id)}
                  className={`
                    flex items-center gap-1.5 md:gap-2 px-3 md:px-4 py-2 md:py-2.5 rounded-xl font-medium transition-all whitespace-nowrap
                    ${selectedCategory === category.id
                      ? 'bg-primary text-white shadow-lg shadow-primary/25'
                      : 'bg-base-2 text-gray-400 hover:bg-base-3 hover:text-white'
                    }
                  `}
                >
                  <Iconify icon={category.icon} size={18} className="md:w-5 md:h-5" />
                  <span className="text-xs md:text-sm">{category.label}</span>
                </button>
              ))}
            </div>
          </div>

          {/* Sort Filter - Mobile */}
          <div className="flex flex-col md:hidden w-full gap-3">
            <div className="flex justify-between">
              <span className="text-gray-400 text-xs whitespace-nowrap">Sırala:</span>
              <div className="text-gray-400 text-xs whitespace-nowrap">
                <span className="text-white font-semibold">{filteredTasks.length}</span> görev
              </div>
            </div>
            <div className="flex items-center gap-2 flex-1 min-w-0">
              <select
                value={sortBy}
                onChange={(e) => handleSortChange(e.target.value as 'highest' | 'lowest' | 'newest')}
                className="bg-base-2 text-white px-2 py-1.5 text-xs rounded-xl border border-white/5 focus:border-primary/50 focus:outline-none transition-colors cursor-pointer flex-1 min-w-0"
              >
                <option value="highest">En Yüksek Ödül</option>
                <option value="lowest">En Düşük Ödül</option>
                <option value="newest">En Yeni</option>
              </select>
            </div>
          </div>

          {/* Sort Filter - Desktop */}
          <div className="hidden md:flex md:items-center md:gap-4 md:flex-shrink-0">
            <div className="flex items-center gap-2">
              <span className="text-gray-400 text-sm whitespace-nowrap">Sırala:</span>
              <select
                value={sortBy}
                onChange={(e) => handleSortChange(e.target.value as 'highest' | 'lowest' | 'newest')}
                className="bg-base-2 text-white px-4 py-2 text-sm rounded-xl border border-white/5 focus:border-primary/50 focus:outline-none transition-colors cursor-pointer min-w-[180px]"
              >
                <option value="highest">En Yüksek Ödül</option>
                <option value="lowest">En Düşük Ödül</option>
                <option value="newest">En Yeni</option>
              </select>
            </div>

            <div className="text-gray-400 text-sm whitespace-nowrap">
              <span className="text-white font-semibold">{filteredTasks.length}</span> görev bulundu
            </div>
          </div>
        </div>

        {/* Tasks Grid */}
        {currentData.length > 0 ? (
          <>
            <div className="grid grid-cols-3 sm:grid-cols-5 lg:grid-cols-6 xl:grid-cols-7 gap-3 md:gap-4 mb-8">
              {currentData.map((task: MacroTask) => (
                <div key={task.id} className="group">
                  <Card 
                    variant="gradient" 
                    img={task.image} 
                    padding="md"
                    className="h-full hover:scale-105 transition-transform duration-300 cursor-pointer"
                    onClick={() => handleCardClick(task)}
                  >
                    <CardContent>
                      <h3 className="font-medium text-sm md:text-lg text-white mb-1 group-hover:text-primary transition-colors line-clamp-1">
                        {task.title}
                      </h3>
                      <p className="text-xs md:text-sm text-white/70 mb-2 md:mb-3 line-clamp-1">{task.subtitle}</p>
                      <div className="flex items-center justify-between">
                        <div className="text-sm md:text-xl font-semibold text-white">{task.amount}</div>
                        <Iconify 
                          icon="mdi:arrow-right" 
                          size={20}
                          className="text-white/50 group-hover:text-primary group-hover:translate-x-1 transition-all md:text-2xl"
                        />
                      </div>
                    </CardContent>
                  </Card>
                </div>
              ))}
            </div>

            {/* Pagination */}
            <Pagination
              currentPage={currentPage}
              totalPages={totalPages}
              onPageChange={goToPage}
              className="mb-8"
            />
          </>
        ) : (
          <div className="text-center py-16">
            <Iconify icon="mdi:file-document-outline" size={64} className="text-gray-600 mx-auto mb-4" />
            <p className="text-gray-400 text-lg">Bu kategoride görev bulunamadı</p>
          </div>
        )}
      </div>
      
      {/* Task Detail Modal */}
      {selectedTask && (
        <TaskDetailModal
          isOpen={isModalOpen}
          onClose={handleModalClose}
          task={selectedTask}
        />
      )}
      <Footer />
    </div>
  );
}
