'use client'

import React, { useState } from 'react'
import Image from 'next/image'
import { cn } from '@/lib/utils'
import { Icon, Iconify } from '@/components/ui'

export interface ChatMessage {
  id: string
  sender: string
  avatar: string
  message: string
  time: string
  isOnline?: boolean
}

export interface LiveSupportProps {
  /** Chat başlığı */
  title?: string
  /** Online kullanıcı sayısı */
  onlineCount?: number
  /** Mesajlar listesi */
  messages?: ChatMessage[]
  /** Yeni mesaj gönderme handler'ı */
  onSendMessage?: (message: string) => void
  /** Chat kapatma handler'ı */
  onClose?: () => void
  /** Ek CSS sınıfları */
  className?: string
  /** Chat açık mı kapalı mı */
  isOpen?: boolean
}
/**
 * LiveSupport Component
 * Canlı destek chat sistemi
 * WhatsApp benzeri arayüz ile kullanıcı dostu tasarım
 */
const LiveSupport: React.FC<LiveSupportProps> = ({
  title = "Canlı Sohbet",
  onlineCount = 354,
  messages = [
    {
      id: '1',
      sender: 'Emma Collins',
      avatar: '/assets/images/pp1.png',
      message: 'Spotify\'da 3dk Manifest şarkısını dinle kazan etkinliği başlamış 🎵 Kaçırmayın, ödüller çok iyi!',
      time: '12:21',
      isOnline: true
    },
    {
      id: '2',
      sender: 'Sophia Reed',
      avatar: '/assets/images/pp1.png',
      message: 'Arkadaşlar Instagram\'da yeni takip kazan kampanyası açıldı 📱 Ödülü efsane, hemen deneyin!',
      time: '12:24',
      isOnline: true
    },
    {
      id: '3',
      sender: 'Lily Turner',
      avatar: '/assets/images/pp1.png',
      message: 'X\'te #LikeVeRetweet kampanyası var 👍 Katılanlara sürpriz hediye gönderiyorlar, şansı kaçırmayın.',
      time: '12:29',
      isOnline: false
    },
    {
      id: '4',
      sender: 'Emma Collins',
      avatar: '/assets/images/pp1.png',
      message: 'TikTok\'ta #Challenge etkinliği var 🎬 Katılanlara hediye kutusu geliyor, eğlenceli de görünüyor hemen yapın!',
      time: '12:34',
      isOnline: true
    },
    {
      id: '5',
      sender: 'Ethan Brooks',
      avatar: '/assets/images/pp1.png',
      message: 'YouTube\'da yorum yap kazan etkinliği başlamış 💬 Abone ol + yorum yap, hediye çekilişine giriyorsun!',
      time: '12:34',
      isOnline: false
    }
  ],
  onSendMessage,
  className,
  isOpen = true
}) => {
  const [newMessage, setNewMessage] = useState('')
  const [isClosed, setIsClosed] = useState(false)
  const [isAnimating, setIsAnimating] = useState(false)

  const handleToggle = () => {
    if (isAnimating) return // Animasyon sırasında tıklamayı engelle
    setIsAnimating(true)
    setIsClosed(!isClosed)
    
  // Animasyon tamamlandıktan sonra animating state'ini resetle
    setTimeout(() => {
      setIsAnimating(false)
    }, 300)
  }

  const handleSendMessage = () => {
    if (newMessage.trim() && onSendMessage) {
      onSendMessage(newMessage.trim())
      setNewMessage('')
    }
  }

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault()
      handleSendMessage()
    }
  }

  if (!isOpen) {
    return null
  }

  return (
    <div
      className={cn(
        'z-10  pt-17 h-screen',
        'bg-base-1 border-l border-primary/20',
        'flex flex-col overflow-hidden shadow-2xl',
        isClosed ? ' w-[80px] ' : 'w-[350px]',
        className
      )}
    >
      {/* Header */}
      <div className="flex items-center justify-between p-4 ">
        <button
          onClick={handleToggle}
          className={cn('flex cursor-pointer items-center gap-2 text-white hover:text-gray-300 transition-colors',
            isClosed && 'pl-1'
          )}
        >
          <div className={cn(
            'w-8 h-8  rounded-lg bg-white/10  flex items-center justify-center'
          )}>
            <div className={cn(
            'transition-transform  flex items-center justify-center duration-800 ease-in-out',
            isClosed && 'rotate-180',)} >
            <Iconify icon="solar:alt-arrow-right-linear" size="md" /></div>
          </div>
          <span className="text-sm font-medium">
            {isClosed ? '' : 'Canlı Sohbeti Kapat'}
          </span>
        </button>
      </div>

      {/* Chat Info */}
      {!isClosed && ( <div className="flex items-center gap-3 py-1 px-3 ">
        <div className="relative w-12 h-10 flex items-center justify-center  bg-base-3 rounded-lg">
          <Icon name="gridicons:chat" size="lg" />
        </div>
        <div className='w-full  flex  justify-between items-center'>
          <h3 className="text-white font-semibold">{title}</h3>
          <div className="flex items-center gap-1 text-sm">
            <div className="w-2 h-2 bg-[#03FF9A] rounded-full animate-pulse"></div>
            <span className="text-[13px] text-white/70">
              {onlineCount} Çevrimiçi
            </span>
          </div>
        </div>
      </div> )}
     

{/* Messages */}
<div className="flex-1  overflow-y-auto p-4 space-y-4 scrollbar-hide">
  {messages.map((message) => (
    <div key={message.id} className="flex items-end gap-2">
      {/* Avatar */}
      <div className="relative flex-shrink-0">
        <Image
          src={message.avatar}
          alt={message.sender}
          width={40}
          height={40}
          className="w-10 h-10 rounded-full object-cover"
        />
        {message.isOnline && (
          <div className="absolute -bottom-0 -right-0 w-3 h-3 bg-green-500 rounded-full border-2 border-base-1"></div>
        )}
      </div>
      {/* Message Content */}
      {!isClosed && (
        <div className="flex-1 min-w-0">
          <div className="flex items-center gap-2 mb-1">
            <span className="text-white font-medium text-sm">
              {message.sender}
            </span>
          </div>
          <div className="bg-base-3 relative rounded-2xl rounded-bl-none p-3">
            <span className='absolute -left-2.5 -bottom-1'>
              <svg width="27" height="28" viewBox="0 0 27 28" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M7.09989 21.8499C11.6643 16.8499 10.65 0.849945 10.65 0.849945L26.8789 20.3499C20.7931 25.3499 11.6643 28.8499 -0.000267029 27.3499C-0.000267029 27.3499 2.5355 26.8499 7.09989 21.8499Z" fill="#211D2F"/>
              </svg>
            </span>
            <div className='flex flex-col '>
              <span className="text-gray-300 text-sm leading-relaxed">
                {message.message}
              </span>
              <span className=" flex justify-end text-gray-400 w-full text-xs">
                {message.time}
              </span>
            </div>
          </div>
        </div>
      )}
    </div>
  ))}
</div>
     

      {/* Message Input */}
            {!isClosed &&(<div className="p-4 border-t border-base-3">
        <div className="flex items-center gap-2">
          <div className="flex-1 relative">
            <input
              type="text"
              value={newMessage}
              onChange={(e) => setNewMessage(e.target.value)}
              onKeyPress={handleKeyPress}
              placeholder="Mesajınız"
              className={cn(
                'w-full px-4 py-3 pr-12 rounded-full',
                'bg-base-2 border border-base-3',
                'text-white placeholder-gray-400',
                'focus:outline-none focus:border-primary/50',
                'transition-colors'
              )}
            />
            <button
              onClick={() => {}} // Emoji picker functionality
              className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-white transition-colors"
            >
              <Iconify icon="solar:smile-circle-linear" size="md" />
            </button>
          </div>
          <button
            onClick={handleSendMessage}
            disabled={!newMessage.trim()}
            className={cn(
              'w-10 h-10 rounded-full flex items-center justify-center',
              'bg-primary hover:bg-primary/80 transition-colors',
              'disabled:opacity-50 disabled:cursor-not-allowed',
              'text-white'
            )}
          >
            <Iconify icon="solar:plain-2-linear" size="md" />
          </button>
        </div>
      </div>  )}
      
    </div>
  )
}

export { LiveSupport }
export default LiveSupport