/**
 * Featured Tasks Section Component
 * SOLID Principles:
 * - Single Responsibility: Display featured tasks in an auto-scrolling carousel
 * - Open/Closed: Configurable through props
 * - Separation of Concerns: Data generation handled by data layer
 */

'use client'

import { useState, useEffect, useRef } from 'react'
import { Card, CardContent, Icon } from "@/components/ui"
import { Swiper, SwiperSlide } from 'swiper/react'
import Image from "next/image"
import type { FeaturedTaskCardData } from '@/data'
import { getInitialFeaturedTasks } from '@/data'
import type { Swiper as SwiperType } from 'swiper'

// Import Swiper styles
import 'swiper/css'

interface FeaturedTasksSectionProps {
  initialTaskCount?: number
}

export default function FeaturedTasksSection({ 
  initialTaskCount = 10
}: FeaturedTasksSectionProps) {
  const [tasks, setTasks] = useState<FeaturedTaskCardData[]>([])
  const swiperRef = useRef<SwiperType | null>(null)

  // Initialize tasks on mount
  useEffect(() => {
    const initialTasks = getInitialFeaturedTasks(initialTaskCount)
    setTasks(initialTasks)
    
    // Set swiper to the last slide (rightmost) after tasks are loaded
    setTimeout(() => {
      if (swiperRef.current && initialTasks.length > 0) {
        swiperRef.current.slideTo(initialTasks.length - 1, 0)
      }
    }, 100)
  }, [initialTaskCount])

  // Auto-scroll: Slide left every 10 seconds
  useEffect(() => {
    if (tasks.length === 0) return

    // Wait a bit before starting auto-scroll
    let scrollInterval: NodeJS.Timeout

    const initialDelay = setTimeout(() => {
      scrollInterval = setInterval(() => {
        if (swiperRef.current) {
          const currentIndex = swiperRef.current.activeIndex
          // Slide one card to the left (decrement index by 1)
          // If at the beginning, wrap to the end
          if (currentIndex > 0) {
            swiperRef.current.slideTo(currentIndex - 1, 1200)
          } else {
            swiperRef.current.slideTo(tasks.length - 1, 1200)
          }
        }
      }, 10000) // Every 10 seconds
    }, 2000) // Start after 2 seconds

    return () => {
      clearTimeout(initialDelay)
      if (scrollInterval) clearInterval(scrollInterval)
    }
  }, [tasks.length])

  return (
    <div className="px-4 overflow-hidden"> 
      <Swiper
        modules={[]}
        spaceBetween={10}
        slidesPerView="auto"
        allowTouchMove={false}
        simulateTouch={false}
        speed={1200}
        cssMode={false}
        className="!overflow-visible"
        onSwiper={(swiper) => {
          swiperRef.current = swiper
        }}
      >
        {tasks.map((task) => (
          <SwiperSlide key={task.id} className="!w-28 md:!w-48">
            <Card variant="gradient" className="md:p-3">
              <CardContent>
                <div className="flex items-center justify-center space-x-1 md:space-x-1.5 mb-1.5 md:mb-3">
                  <Icon size='sm' className="md:text-xl" name='solar:money-bag-bold' />
                  <h3 className="text-bold text-[10px] md:text-sm truncate">{task.amount} {task.title}</h3>
                </div>
                <svg className="w-full" width={100} height="2" viewBox="0 0 148 1" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <line x1="-14" y1="0.5" x2="161" y2="0.500015" stroke="url(#paint0_radial_1010_7445)"/>
                  <defs>
                    <radialGradient id="paint0_radial_1010_7445" cx="0" cy="0" r="1" gradientUnits="userSpaceOnUse" gradientTransform="translate(75.8649 1.00001) rotate(90.0003) scale(0.702899 84.5754)">
                      <stop offset="0.264423" stopColor="#8570FF"/>
                      <stop offset="1" stopColor="#8570FF" stopOpacity="0"/>
                    </radialGradient>
                  </defs>
                </svg>
                <div className="p-0.5 md:p-1.5 relative space-x-1 md:space-x-1.5 flex flex-row justify-center items-center">
                  <div className="relative flex-shrink-0">
                    <Image
                      src={task.userImage}
                      alt={task.userName}
                      width={100}
                      height={100}
                      className="w-6 h-6 md:w-8 md:h-8 rounded-full"
                    />
                    <span className="-right-0.5 border-2 border-base-2 bottom-0.5 absolute w-2 h-2 md:w-2.5 md:h-2.5 bg-green-400 rounded-full"/>
                  </div>
                  <div className="-space-y-1 min-w-0 flex-1">
                    <div className="text-[10px] md:text-sm font-normal text-white truncate">{task.userName}</div>
                    <div className="text-[8px] md:text-xs font-normal text-white/70 truncate">{task.userNickname}</div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </SwiperSlide>
        ))}
      </Swiper>
    </div>
  )
}