'use client'

import { useState } from "react";
import { useRouter } from "next/navigation";
import { Card, CardContent, Icon, Iconify } from "@/components/ui";
import { TaskDetailModal } from "@/components/sections";
import type { IconName } from "@/lib/icons";
import { Swiper, SwiperSlide } from 'swiper/react';
import { Navigation } from 'swiper/modules';
import type { Swiper as SwiperType } from 'swiper';
import type { OfferCardData } from "@/data/offers";

// Import Swiper styles
import 'swiper/css';
import 'swiper/css/navigation';

interface FeaturedOffersProps {
  title?: string ;
  iconName?: string;
  offers?: OfferCardData[];
  cardVariant?: 'default' | 'elevated' | 'bordered' | 'gradient' | 'gradient2';
  all_item?: boolean;
  enableModal?: boolean; // Modal açma/kapama kontrolü
}

export default function FeaturedOffers({
  title = "Öne Çıkan Teklifler",
  iconName = "mdi:fire",
  offers = [],
  cardVariant = "gradient",
  all_item = false,
  enableModal = false // Default olarak modal kapalı
}: FeaturedOffersProps) {
  const [swiperInstance, setSwiperInstance] = useState<SwiperType | null>(null);
  const router = useRouter();
  
  // Modal state
  const [selectedOffer, setSelectedOffer] = useState<OfferCardData | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);

  // Modal açma fonksiyonu - sadece enableModal true ise çalışır
  const handleCardClick = (offer: OfferCardData) => {
    if (!enableModal) return; // Modal kapalıysa hiçbir şey yapma
    
    setSelectedOffer(offer);
    setIsModalOpen(true);
  };

  const handleModalClose = () => {
    setIsModalOpen(false);
    setSelectedOffer(null);
  };

  return (
    <div className="px-4 pt-2 overflow-hidden flex flex-col">
      <div className="flex items-center justify-between space-x-1">
        <div className="flex gap-1.5 md:gap-2 justify-center items-center">
        <Icon size='lg' className="md:text-2xl" name={iconName as IconName} />
        <h2 className="text-sm md:text-lg font-semibold">{title}</h2>
        </div>
       
        <div className="flex items-center gap-1.5 md:gap-2">
         
          {all_item && (<button  onClick={() => router.push('/macro-tasks')} className="text-sm cursor-pointer hover:text-primary transition-colors">
            Tümünü Gör
          </button>)}
          
          <button
            onClick={() => swiperInstance?.slidePrev()}
            className="w-7 h-7 md:w-9 md:h-9 flex items-center justify-center bg-base-2 hover:bg-base-3 rounded-lg transition-colors"
            aria-label="Önceki"
          >
            <Iconify icon="solar:alt-arrow-left-bold" size={16} className="text-white md:text-lg" />
          </button>
          <button
            onClick={() => swiperInstance?.slideNext()}
            className="w-7 h-7 md:w-9 md:h-9 flex items-center justify-center bg-base-2 hover:bg-base-3 rounded-lg transition-colors"
            aria-label="Sonraki"
          >
            <Iconify icon="solar:alt-arrow-right-bold" size={16} className="text-white md:text-lg" />
          </button>
        </div>
       
      </div>
      
      <div className="mt-2 md:mt-3">
        <Swiper
          modules={[Navigation]}
          spaceBetween={10}
          slidesPerView="auto"
          onSwiper={(swiper) => setSwiperInstance(swiper)}
          className="!overflow-visible"
        >
          {offers.map((offer) => (
            <SwiperSlide key={offer.id} className="!w-28 md:!w-40">
              <Card 
                variant={cardVariant} 
                img={offer.image} 
                category={offer.category}
                className={`md:p-2 ${enableModal ? 'cursor-pointer' : 'cursor-default'}`}
                onClick={() => handleCardClick(offer)}
              >
                <CardContent>
                  {cardVariant === 'gradient2' ? (
                    // Gradient2 için yeni düzen - Logo, başlık ve yıldızlar
                    <div className="flex flex-col items-center justify-center gap-1.5 md:gap-2 relative z-10">
                      <h3 className="font-medium text-xs md:text-base text-white text-center line-clamp-1">{offer.title}</h3>
                      <div className="flex items-center gap-0.5">
                        {[1, 2, 3, 4, 5].map((star) => (
                          <Iconify 
                            key={star} 
                            icon="material-symbols:star" 
                            className={star <= (offer.rating || 5) ? "text-yellow-400" : "text-gray-600"} 
                            size={12}
                          />
                        ))}
                      </div>
                    </div>
                  ) : (
                    // Diğer variantlar için eski düzen
                    <>
                      <h3 className="font-medium text-xs md:text-base text-white mb-0.5 md:mb-1 line-clamp-1">{offer.title}</h3>
                      <p className="text-[10px] md:text-xs text-white/70 mb-1 md:mb-2 line-clamp-1">{offer.subtitle}</p>
                      <div className="text-xs md:text-base font-semibold text-white">{offer.amount}</div>
                    </>
                  )}
                </CardContent>
              </Card>
            </SwiperSlide>
          ))}
        </Swiper>
      </div>

      {/* Task Detail Modal */}
      <TaskDetailModal 
        isOpen={isModalOpen} 
        onClose={handleModalClose} 
        task={selectedOffer}
      />
    </div>
  );
}