'use strict';

const MagicString = require('magic-string');
const compilerSfc = require('@vue/compiler-sfc');
const parser = require('@babel/parser');
const estreeWalker = require('estree-walker');

function _interopDefaultCompat (e) { return e && typeof e === 'object' && 'default' in e ? e.default : e; }

const MagicString__default = /*#__PURE__*/_interopDefaultCompat(MagicString);

const REGEX_LANG_TS = /[mc]?tsx?$/;
const REGEX_LANG_JSX = /.?[jt]sx$/;
const REGEX_VUE = /^vue|\.vue$/;
const REGEX_ACCEPTABLE_LANG = /\.(vue|[mc]?[jt]sx?)$/;
const isTS = (lang) => REGEX_LANG_TS.test(lang);
const isJSX = (lang) => REGEX_LANG_JSX.test(lang);
const isVUE = (filename) => REGEX_VUE.test(filename);
const isAcceptableLang = (filename) => REGEX_ACCEPTABLE_LANG.test(filename);

function entries(obj) {
  return Object.entries(obj);
}

function ensureImport(code, importPackages, offset) {
  for (const [source, packages] of entries(importPackages)) {
    const prependCode = `
;import { ${packages.map((p) => `${p.id} as ${p.alias}`).join(", ")} } from '${source}'
;`;
    code.prependRight(offset, `${prependCode}
`);
  }
  return code;
}

function getBabelParsePlugins(lang) {
  const plugins = [];
  if (isTS(lang))
    plugins.push("typescript");
  if (isJSX(lang))
    plugins.push("jsx");
  return plugins;
}
function babelParse(code, lang) {
  const options = {
    plugins: getBabelParsePlugins(lang),
    sourceType: "module"
  };
  return parser.parse(code, options);
}
function parseSFC(code, filename) {
  const sfc = compilerSfc.parse(code, {
    filename
  });
  const { descriptor } = sfc;
  const scriptLang = descriptor.script?.lang ?? "js";
  const scriptSetupLang = descriptor.scriptSetup?.lang ?? "js";
  if (descriptor.script && descriptor.scriptSetup && scriptLang !== scriptSetupLang)
    throw new Error(`[vue-devtools] ${filename} <script> and <script setup> must use the same language`);
  const lang = scriptLang || scriptSetupLang;
  return {
    sfc,
    script: descriptor.script,
    scriptSetup: descriptor.scriptSetup,
    scriptLocation: {
      start: descriptor.script?.loc.start.offset ?? 0,
      end: descriptor.script?.loc.end.offset ?? 0
    },
    scriptSetupLocation: {
      start: descriptor.scriptSetup?.loc.start.offset ?? 0,
      end: descriptor.scriptSetup?.loc.end.offset ?? 0
    },
    getScriptAST() {
      if (!descriptor.script)
        return;
      return babelParse(descriptor.script.content, lang);
    },
    getScriptSetupAST() {
      if (!descriptor.scriptSetup)
        return;
      return babelParse(descriptor.scriptSetup.content, lang);
    }
  };
}

async function walkAST(node, handlers) {
  return estreeWalker.walk(node, handlers);
}
function getNodeEndAndFilterReturn(node) {
  let end = node.end ?? 0;
  for (const stmt of node.body) {
    if (stmt.type === "ReturnStatement") {
      end = stmt.start ?? 0;
      break;
    }
  }
  return end;
}
function isObjectFn(node) {
  return node.type === "ObjectMethod" && node.kind === "method" || node.type === "ObjectProperty" && (node.value.type === "FunctionExpression" || node.value.type === "ArrowFunctionExpression") && node.value.body.type === "BlockStatement";
}
function getObjectFnBodyLocation(node) {
  if (node.type === "ObjectMethod") {
    return {
      start: node.body.start ?? 0,
      end: getNodeEndAndFilterReturn(node.body)
    };
  }
  const body = node.value;
  if (body.type !== "ArrowFunctionExpression" && body.type !== "FunctionExpression")
    return null;
  return {
    start: body.body.start ?? 0,
    end: getNodeEndAndFilterReturn(body.body)
  };
}
function isCallOf(node, name) {
  return node.type === "CallExpression" && node.callee.type === "Identifier" && node.callee.name === name;
}

const SETUP_FN = "setup";
const DEFINE_COMPONENT = "defineComponent";

function isSetupFn(node) {
  return isObjectFn(node) && node.key.name === SETUP_FN;
}
function analyzeVueSFC(code, filename) {
  const {
    scriptSetup,
    scriptSetupLocation,
    script,
    scriptLocation,
    getScriptAST
  } = parseSFC(code, filename);
  if (!scriptSetup && !script)
    return null;
  if (!scriptSetup && script) {
    const offset = scriptLocation.start;
    const ast = getScriptAST();
    let location = null;
    walkAST(ast, {
      enter(node) {
        if (isSetupFn(node)) {
          const loc = getObjectFnBodyLocation(node);
          location = loc ? {
            start: offset,
            end: offset + loc.end
          } : null;
          this.skip();
        }
      }
    });
    return location;
  }
  return scriptSetup ? scriptSetupLocation : scriptLocation;
}
function analyzeScriptFile(code, lang) {
  if (!code.trim().length || !code.includes(DEFINE_COMPONENT))
    return null;
  const location = [];
  const ast = babelParse(code, lang);
  walkAST(ast, {
    enter(node) {
      if (isCallOf(node, DEFINE_COMPONENT) && node.arguments[0] && node.arguments[0].type === "ObjectExpression") {
        for (const stmt of node.arguments[0].properties) {
          if (isSetupFn(stmt)) {
            const loc = getObjectFnBodyLocation(stmt);
            location.push(loc ? {
              start: ast.start ?? 0,
              end: loc.end
            } : null);
            this.skip();
          }
        }
      }
    }
  });
  return location.filter(Boolean);
}

function analyzeByTraceRerender(code, locations) {
  const apiNames = {
    getCurrentInstance: "__VUE_DEVTOOLS_$getCurrentInstance__",
    onRenderTracked: "__VUE_DEVTOOLS_$onRenderTracked__",
    onRenderTriggered: "__VUE_DEVTOOLS_$onRenderTriggered__"
  };
  const injectedCodes = {
    onRenderTracked: `
    
;${apiNames.onRenderTracked}((e) => {
      const instance = ${apiNames.getCurrentInstance}()
      window.__VUE_DEVTOOLS_GLOBAL_HOOK__?.emit?.('render:tracked', e, instance)
    });
`,
    onRenderTriggered: `
    
;${apiNames.onRenderTriggered}((e) => {
      const instance = ${apiNames.getCurrentInstance}()
      window.__VUE_DEVTOOLS_GLOBAL_HOOK__?.emit?.('render:triggered', e, instance)
    });
`
  };
  locations.forEach(({ start, end }, idx) => {
    if (idx === 0) {
      code = ensureImport(code, {
        vue: entries(apiNames).map(([id, alias]) => ({
          id,
          alias
        }))
      }, start);
    }
    entries(injectedCodes).forEach(([, appendCode]) => {
      code.prependLeft(end, appendCode);
    });
  });
  return code;
}

const excludePaths = ["node_modules"];
function enableAnalyze(config) {
  return entries(config).some(([, enable]) => enable);
}
function hitPaths(filename, paths) {
  return paths.some((path) => filename.includes(path));
}
const analyzeOptionsDefault = {
  rerenderTrace: true
};
function analyzeCode(code, filename, options) {
  if (!isAcceptableLang(filename) || !enableAnalyze(options) || hitPaths(filename, excludePaths))
    return code;
  let locations = null;
  if (isVUE(filename)) {
    const location = analyzeVueSFC(code, filename);
    if (location)
      locations = [location];
  } else {
    const lang = filename.split(".").pop();
    locations = analyzeScriptFile(code, lang);
  }
  if (!locations || !locations.length)
    return code;
  let ms = new MagicString__default(code);
  if (options.rerenderTrace)
    ms = analyzeByTraceRerender(ms, locations);
  return {
    code: ms.toString(),
    get map() {
      return ms.generateMap({
        source: filename,
        includeContent: true,
        hires: "boundary"
      });
    }
  };
}

exports.analyzeCode = analyzeCode;
exports.analyzeOptionsDefault = analyzeOptionsDefault;
