const typedArrayTypeNames = [
    'Int8Array',
    'Uint8Array',
    'Uint8ClampedArray',
    'Int16Array',
    'Uint16Array',
    'Int32Array',
    'Uint32Array',
    'Float32Array',
    'Float64Array',
    'BigInt64Array',
    'BigUint64Array',
];
function isTypedArrayName(name) {
    return typedArrayTypeNames.includes(name);
}
const objectTypeNames = [
    'Function',
    'Generator',
    'AsyncGenerator',
    'GeneratorFunction',
    'AsyncGeneratorFunction',
    'AsyncFunction',
    'Observable',
    'Array',
    'Buffer',
    'Blob',
    'Object',
    'RegExp',
    'Date',
    'Error',
    'Map',
    'Set',
    'WeakMap',
    'WeakSet',
    'WeakRef',
    'ArrayBuffer',
    'SharedArrayBuffer',
    'DataView',
    'Promise',
    'URL',
    'FormData',
    'URLSearchParams',
    'HTMLElement',
    'NaN',
    ...typedArrayTypeNames,
];
function isObjectTypeName(name) {
    return objectTypeNames.includes(name);
}
const primitiveTypeNames = [
    'null',
    'undefined',
    'string',
    'number',
    'bigint',
    'boolean',
    'symbol',
];
function isPrimitiveTypeName(name) {
    return primitiveTypeNames.includes(name);
}
const assertionTypeDescriptions = [
    'positive number',
    'negative number',
    'Class',
    'string with a number',
    'null or undefined',
    'Iterable',
    'AsyncIterable',
    'native Promise',
    'EnumCase',
    'string with a URL',
    'truthy',
    'falsy',
    'primitive',
    'integer',
    'plain object',
    'TypedArray',
    'array-like',
    'tuple-like',
    'Node.js Stream',
    'infinite number',
    'empty array',
    'non-empty array',
    'empty string',
    'empty string or whitespace',
    'non-empty string',
    'non-empty string and not whitespace',
    'empty object',
    'non-empty object',
    'empty set',
    'non-empty set',
    'empty map',
    'non-empty map',
    'PropertyKey',
    'even integer',
    'odd integer',
    'T',
    'in range',
    'predicate returns truthy for any value',
    'predicate returns truthy for all values',
    'valid length',
    'whitespace string',
    ...objectTypeNames,
    ...primitiveTypeNames,
];
const getObjectType = (value) => {
    const objectTypeName = Object.prototype.toString.call(value).slice(8, -1);
    if (/HTML\w+Element/.test(objectTypeName) && isHtmlElement(value)) {
        return 'HTMLElement';
    }
    if (isObjectTypeName(objectTypeName)) {
        return objectTypeName;
    }
    return undefined;
};
function detect(value) {
    if (value === null) {
        return 'null';
    }
    switch (typeof value) {
        case 'undefined': {
            return 'undefined';
        }
        case 'string': {
            return 'string';
        }
        case 'number': {
            return Number.isNaN(value) ? 'NaN' : 'number';
        }
        case 'boolean': {
            return 'boolean';
        }
        case 'function': {
            return 'Function';
        }
        case 'bigint': {
            return 'bigint';
        }
        case 'symbol': {
            return 'symbol';
        }
        default:
    }
    if (isObservable(value)) {
        return 'Observable';
    }
    if (isArray(value)) {
        return 'Array';
    }
    if (isBuffer(value)) {
        return 'Buffer';
    }
    const tagType = getObjectType(value);
    if (tagType) {
        return tagType;
    }
    if (value instanceof String || value instanceof Boolean || value instanceof Number) {
        throw new TypeError('Please don\'t use object wrappers for primitive types');
    }
    return 'Object';
}
function hasPromiseApi(value) {
    return isFunction(value?.then) && isFunction(value?.catch);
}
const is = Object.assign(detect, {
    all: isAll,
    any: isAny,
    array: isArray,
    arrayBuffer: isArrayBuffer,
    arrayLike: isArrayLike,
    asyncFunction: isAsyncFunction,
    asyncGenerator: isAsyncGenerator,
    asyncGeneratorFunction: isAsyncGeneratorFunction,
    asyncIterable: isAsyncIterable,
    bigint: isBigint,
    bigInt64Array: isBigInt64Array,
    bigUint64Array: isBigUint64Array,
    blob: isBlob,
    boolean: isBoolean,
    boundFunction: isBoundFunction,
    buffer: isBuffer,
    class: isClass,
    /** @deprecated Renamed to `class`. */
    class_: isClass,
    dataView: isDataView,
    date: isDate,
    detect,
    directInstanceOf: isDirectInstanceOf,
    /** @deprecated Renamed to `htmlElement` */
    domElement: isHtmlElement,
    emptyArray: isEmptyArray,
    emptyMap: isEmptyMap,
    emptyObject: isEmptyObject,
    emptySet: isEmptySet,
    emptyString: isEmptyString,
    emptyStringOrWhitespace: isEmptyStringOrWhitespace,
    enumCase: isEnumCase,
    error: isError,
    evenInteger: isEvenInteger,
    falsy: isFalsy,
    float32Array: isFloat32Array,
    float64Array: isFloat64Array,
    formData: isFormData,
    function: isFunction,
    /** @deprecated Renamed to `function`. */
    function_: isFunction,
    generator: isGenerator,
    generatorFunction: isGeneratorFunction,
    htmlElement: isHtmlElement,
    infinite: isInfinite,
    inRange: isInRange,
    int16Array: isInt16Array,
    int32Array: isInt32Array,
    int8Array: isInt8Array,
    integer: isInteger,
    iterable: isIterable,
    map: isMap,
    nan: isNan,
    nativePromise: isNativePromise,
    negativeNumber: isNegativeNumber,
    nodeStream: isNodeStream,
    nonEmptyArray: isNonEmptyArray,
    nonEmptyMap: isNonEmptyMap,
    nonEmptyObject: isNonEmptyObject,
    nonEmptySet: isNonEmptySet,
    nonEmptyString: isNonEmptyString,
    nonEmptyStringAndNotWhitespace: isNonEmptyStringAndNotWhitespace,
    null: isNull,
    /** @deprecated Renamed to `null`. */
    null_: isNull,
    nullOrUndefined: isNullOrUndefined,
    number: isNumber,
    numericString: isNumericString,
    object: isObject,
    observable: isObservable,
    oddInteger: isOddInteger,
    plainObject: isPlainObject,
    positiveNumber: isPositiveNumber,
    primitive: isPrimitive,
    promise: isPromise,
    propertyKey: isPropertyKey,
    regExp: isRegExp,
    safeInteger: isSafeInteger,
    set: isSet,
    sharedArrayBuffer: isSharedArrayBuffer,
    string: isString,
    symbol: isSymbol,
    truthy: isTruthy,
    tupleLike: isTupleLike,
    typedArray: isTypedArray,
    uint16Array: isUint16Array,
    uint32Array: isUint32Array,
    uint8Array: isUint8Array,
    uint8ClampedArray: isUint8ClampedArray,
    undefined: isUndefined,
    urlInstance: isUrlInstance,
    urlSearchParams: isUrlSearchParams,
    urlString: isUrlString,
    validLength: isValidLength,
    weakMap: isWeakMap,
    weakRef: isWeakRef,
    weakSet: isWeakSet,
    whitespaceString: isWhitespaceString,
});
function isAbsoluteMod2(remainder) {
    return (value) => isInteger(value) && Math.abs(value % 2) === remainder;
}
export function isAll(predicate, ...values) {
    return predicateOnArray(Array.prototype.every, predicate, values);
}
export function isAny(predicate, ...values) {
    const predicates = isArray(predicate) ? predicate : [predicate];
    return predicates.some(singlePredicate => predicateOnArray(Array.prototype.some, singlePredicate, values));
}
export function isArray(value, assertion) {
    if (!Array.isArray(value)) {
        return false;
    }
    if (!isFunction(assertion)) {
        return true;
    }
    // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
    return value.every(element => assertion(element));
}
export function isArrayBuffer(value) {
    return getObjectType(value) === 'ArrayBuffer';
}
export function isArrayLike(value) {
    return !isNullOrUndefined(value) && !isFunction(value) && isValidLength(value.length);
}
export function isAsyncFunction(value) {
    return getObjectType(value) === 'AsyncFunction';
}
export function isAsyncGenerator(value) {
    return isAsyncIterable(value) && isFunction(value.next) && isFunction(value.throw);
}
export function isAsyncGeneratorFunction(value) {
    return getObjectType(value) === 'AsyncGeneratorFunction';
}
export function isAsyncIterable(value) {
    return isFunction(value?.[Symbol.asyncIterator]);
}
export function isBigint(value) {
    return typeof value === 'bigint';
}
export function isBigInt64Array(value) {
    return getObjectType(value) === 'BigInt64Array';
}
export function isBigUint64Array(value) {
    return getObjectType(value) === 'BigUint64Array';
}
export function isBlob(value) {
    return getObjectType(value) === 'Blob';
}
export function isBoolean(value) {
    return value === true || value === false;
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isBoundFunction(value) {
    return isFunction(value) && !Object.prototype.hasOwnProperty.call(value, 'prototype');
}
export function isBuffer(value) {
    // eslint-disable-next-line @typescript-eslint/no-unsafe-return, @typescript-eslint/no-unsafe-call
    return value?.constructor?.isBuffer?.(value) ?? false;
}
export function isClass(value) {
    return isFunction(value) && value.toString().startsWith('class ');
}
export function isDataView(value) {
    return getObjectType(value) === 'DataView';
}
export function isDate(value) {
    return getObjectType(value) === 'Date';
}
export function isDirectInstanceOf(instance, class_) {
    return Object.getPrototypeOf(instance) === class_.prototype;
}
export function isEmptyArray(value) {
    return isArray(value) && value.length === 0;
}
export function isEmptyMap(value) {
    return isMap(value) && value.size === 0;
}
export function isEmptyObject(value) {
    return isObject(value) && !isMap(value) && !isSet(value) && Object.keys(value).length === 0;
}
export function isEmptySet(value) {
    return isSet(value) && value.size === 0;
}
export function isEmptyString(value) {
    return isString(value) && value.length === 0;
}
export function isEmptyStringOrWhitespace(value) {
    return isEmptyString(value) || isWhitespaceString(value);
}
export function isEnumCase(value, targetEnum) {
    // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
    return Object.values(targetEnum).includes(value);
}
export function isError(value) {
    return getObjectType(value) === 'Error';
}
export function isEvenInteger(value) {
    return isAbsoluteMod2(0)(value);
}
// Example: `is.falsy = (value: unknown): value is (not true | 0 | '' | undefined | null) => Boolean(value);`
export function isFalsy(value) {
    return !value;
}
export function isFloat32Array(value) {
    return getObjectType(value) === 'Float32Array';
}
export function isFloat64Array(value) {
    return getObjectType(value) === 'Float64Array';
}
export function isFormData(value) {
    return getObjectType(value) === 'FormData';
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isFunction(value) {
    return typeof value === 'function';
}
export function isGenerator(value) {
    return isIterable(value) && isFunction(value?.next) && isFunction(value?.throw);
}
export function isGeneratorFunction(value) {
    return getObjectType(value) === 'GeneratorFunction';
}
// eslint-disable-next-line @typescript-eslint/naming-convention
const NODE_TYPE_ELEMENT = 1;
// eslint-disable-next-line @typescript-eslint/naming-convention
const DOM_PROPERTIES_TO_CHECK = [
    'innerHTML',
    'ownerDocument',
    'style',
    'attributes',
    'nodeValue',
];
export function isHtmlElement(value) {
    return isObject(value)
        && value.nodeType === NODE_TYPE_ELEMENT
        && isString(value.nodeName)
        && !isPlainObject(value)
        && DOM_PROPERTIES_TO_CHECK.every(property => property in value);
}
export function isInfinite(value) {
    return value === Number.POSITIVE_INFINITY || value === Number.NEGATIVE_INFINITY;
}
export function isInRange(value, range) {
    if (isNumber(range)) {
        return value >= Math.min(0, range) && value <= Math.max(range, 0);
    }
    if (isArray(range) && range.length === 2) {
        return value >= Math.min(...range) && value <= Math.max(...range);
    }
    throw new TypeError(`Invalid range: ${JSON.stringify(range)}`);
}
export function isInt16Array(value) {
    return getObjectType(value) === 'Int16Array';
}
export function isInt32Array(value) {
    return getObjectType(value) === 'Int32Array';
}
export function isInt8Array(value) {
    return getObjectType(value) === 'Int8Array';
}
export function isInteger(value) {
    return Number.isInteger(value);
}
export function isIterable(value) {
    return isFunction(value?.[Symbol.iterator]);
}
export function isMap(value) {
    return getObjectType(value) === 'Map';
}
export function isNan(value) {
    return Number.isNaN(value);
}
export function isNativePromise(value) {
    return getObjectType(value) === 'Promise';
}
export function isNegativeNumber(value) {
    return isNumber(value) && value < 0;
}
export function isNodeStream(value) {
    return isObject(value) && isFunction(value.pipe) && !isObservable(value);
}
export function isNonEmptyArray(value) {
    return isArray(value) && value.length > 0;
}
export function isNonEmptyMap(value) {
    return isMap(value) && value.size > 0;
}
// TODO: Use `not` operator here to remove `Map` and `Set` from type guard:
// - https://github.com/Microsoft/TypeScript/pull/29317
export function isNonEmptyObject(value) {
    return isObject(value) && !isMap(value) && !isSet(value) && Object.keys(value).length > 0;
}
export function isNonEmptySet(value) {
    return isSet(value) && value.size > 0;
}
// TODO: Use `not ''` when the `not` operator is available.
export function isNonEmptyString(value) {
    return isString(value) && value.length > 0;
}
// TODO: Use `not ''` when the `not` operator is available.
export function isNonEmptyStringAndNotWhitespace(value) {
    return isString(value) && !isEmptyStringOrWhitespace(value);
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isNull(value) {
    return value === null;
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isNullOrUndefined(value) {
    return isNull(value) || isUndefined(value);
}
export function isNumber(value) {
    return typeof value === 'number' && !Number.isNaN(value);
}
export function isNumericString(value) {
    return isString(value) && !isEmptyStringOrWhitespace(value) && !Number.isNaN(Number(value));
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isObject(value) {
    return !isNull(value) && (typeof value === 'object' || isFunction(value));
}
export function isObservable(value) {
    if (!value) {
        return false;
    }
    // eslint-disable-next-line no-use-extend-native/no-use-extend-native, @typescript-eslint/no-unsafe-call
    if (value === value[Symbol.observable]?.()) {
        return true;
    }
    // eslint-disable-next-line @typescript-eslint/no-unsafe-call
    if (value === value['@@observable']?.()) {
        return true;
    }
    return false;
}
export function isOddInteger(value) {
    return isAbsoluteMod2(1)(value);
}
export function isPlainObject(value) {
    // From: https://github.com/sindresorhus/is-plain-obj/blob/main/index.js
    if (typeof value !== 'object' || value === null) {
        return false;
    }
    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
    const prototype = Object.getPrototypeOf(value);
    return (prototype === null || prototype === Object.prototype || Object.getPrototypeOf(prototype) === null) && !(Symbol.toStringTag in value) && !(Symbol.iterator in value);
}
export function isPositiveNumber(value) {
    return isNumber(value) && value > 0;
}
export function isPrimitive(value) {
    return isNull(value) || isPrimitiveTypeName(typeof value);
}
export function isPromise(value) {
    return isNativePromise(value) || hasPromiseApi(value);
}
// `PropertyKey` is any value that can be used as an object key (string, number, or symbol)
export function isPropertyKey(value) {
    return isAny([isString, isNumber, isSymbol], value);
}
export function isRegExp(value) {
    return getObjectType(value) === 'RegExp';
}
export function isSafeInteger(value) {
    return Number.isSafeInteger(value);
}
export function isSet(value) {
    return getObjectType(value) === 'Set';
}
export function isSharedArrayBuffer(value) {
    return getObjectType(value) === 'SharedArrayBuffer';
}
export function isString(value) {
    return typeof value === 'string';
}
export function isSymbol(value) {
    return typeof value === 'symbol';
}
// Example: `is.truthy = (value: unknown): value is (not false | not 0 | not '' | not undefined | not null) => Boolean(value);`
// eslint-disable-next-line unicorn/prefer-native-coercion-functions
export function isTruthy(value) {
    return Boolean(value);
}
export function isTupleLike(value, guards) {
    if (isArray(guards) && isArray(value) && guards.length === value.length) {
        return guards.every((guard, index) => guard(value[index]));
    }
    return false;
}
export function isTypedArray(value) {
    return isTypedArrayName(getObjectType(value));
}
export function isUint16Array(value) {
    return getObjectType(value) === 'Uint16Array';
}
export function isUint32Array(value) {
    return getObjectType(value) === 'Uint32Array';
}
export function isUint8Array(value) {
    return getObjectType(value) === 'Uint8Array';
}
export function isUint8ClampedArray(value) {
    return getObjectType(value) === 'Uint8ClampedArray';
}
export function isUndefined(value) {
    return value === undefined;
}
export function isUrlInstance(value) {
    return getObjectType(value) === 'URL';
}
// eslint-disable-next-line unicorn/prevent-abbreviations
export function isUrlSearchParams(value) {
    return getObjectType(value) === 'URLSearchParams';
}
export function isUrlString(value) {
    if (!isString(value)) {
        return false;
    }
    try {
        new URL(value); // eslint-disable-line no-new
        return true;
    }
    catch {
        return false;
    }
}
export function isValidLength(value) {
    return isSafeInteger(value) && value >= 0;
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isWeakMap(value) {
    return getObjectType(value) === 'WeakMap';
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isWeakRef(value) {
    return getObjectType(value) === 'WeakRef';
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function isWeakSet(value) {
    return getObjectType(value) === 'WeakSet';
}
export function isWhitespaceString(value) {
    return isString(value) && /^\s+$/.test(value);
}
function predicateOnArray(method, predicate, values) {
    if (!isFunction(predicate)) {
        throw new TypeError(`Invalid predicate: ${JSON.stringify(predicate)}`);
    }
    if (values.length === 0) {
        throw new TypeError('Invalid number of values');
    }
    return method.call(values, predicate);
}
function typeErrorMessage(description, value) {
    return `Expected value which is \`${description}\`, received value of type \`${is(value)}\`.`;
}
function unique(values) {
    // eslint-disable-next-line unicorn/prefer-spread
    return Array.from(new Set(values));
}
const andFormatter = new Intl.ListFormat('en', { style: 'long', type: 'conjunction' });
const orFormatter = new Intl.ListFormat('en', { style: 'long', type: 'disjunction' });
function typeErrorMessageMultipleValues(expectedType, values) {
    const uniqueExpectedTypes = unique((isArray(expectedType) ? expectedType : [expectedType]).map(value => `\`${value}\``));
    const uniqueValueTypes = unique(values.map(value => `\`${is(value)}\``));
    return `Expected values which are ${orFormatter.format(uniqueExpectedTypes)}. Received values of type${uniqueValueTypes.length > 1 ? 's' : ''} ${andFormatter.format(uniqueValueTypes)}.`;
}
export const assert = {
    all: assertAll,
    any: assertAny,
    array: assertArray,
    arrayBuffer: assertArrayBuffer,
    arrayLike: assertArrayLike,
    asyncFunction: assertAsyncFunction,
    asyncGenerator: assertAsyncGenerator,
    asyncGeneratorFunction: assertAsyncGeneratorFunction,
    asyncIterable: assertAsyncIterable,
    bigint: assertBigint,
    bigInt64Array: assertBigInt64Array,
    bigUint64Array: assertBigUint64Array,
    blob: assertBlob,
    boolean: assertBoolean,
    boundFunction: assertBoundFunction,
    buffer: assertBuffer,
    class: assertClass,
    class_: assertClass,
    dataView: assertDataView,
    date: assertDate,
    directInstanceOf: assertDirectInstanceOf,
    domElement: assertHtmlElement,
    emptyArray: assertEmptyArray,
    emptyMap: assertEmptyMap,
    emptyObject: assertEmptyObject,
    emptySet: assertEmptySet,
    emptyString: assertEmptyString,
    emptyStringOrWhitespace: assertEmptyStringOrWhitespace,
    enumCase: assertEnumCase,
    error: assertError,
    evenInteger: assertEvenInteger,
    falsy: assertFalsy,
    float32Array: assertFloat32Array,
    float64Array: assertFloat64Array,
    formData: assertFormData,
    function: assertFunction,
    function_: assertFunction,
    generator: assertGenerator,
    generatorFunction: assertGeneratorFunction,
    htmlElement: assertHtmlElement,
    infinite: assertInfinite,
    inRange: assertInRange,
    int16Array: assertInt16Array,
    int32Array: assertInt32Array,
    int8Array: assertInt8Array,
    integer: assertInteger,
    iterable: assertIterable,
    map: assertMap,
    nan: assertNan,
    nativePromise: assertNativePromise,
    negativeNumber: assertNegativeNumber,
    nodeStream: assertNodeStream,
    nonEmptyArray: assertNonEmptyArray,
    nonEmptyMap: assertNonEmptyMap,
    nonEmptyObject: assertNonEmptyObject,
    nonEmptySet: assertNonEmptySet,
    nonEmptyString: assertNonEmptyString,
    nonEmptyStringAndNotWhitespace: assertNonEmptyStringAndNotWhitespace,
    null: assertNull,
    null_: assertNull,
    nullOrUndefined: assertNullOrUndefined,
    number: assertNumber,
    numericString: assertNumericString,
    object: assertObject,
    observable: assertObservable,
    oddInteger: assertOddInteger,
    plainObject: assertPlainObject,
    positiveNumber: assertPositiveNumber,
    primitive: assertPrimitive,
    promise: assertPromise,
    propertyKey: assertPropertyKey,
    regExp: assertRegExp,
    safeInteger: assertSafeInteger,
    set: assertSet,
    sharedArrayBuffer: assertSharedArrayBuffer,
    string: assertString,
    symbol: assertSymbol,
    truthy: assertTruthy,
    tupleLike: assertTupleLike,
    typedArray: assertTypedArray,
    uint16Array: assertUint16Array,
    uint32Array: assertUint32Array,
    uint8Array: assertUint8Array,
    uint8ClampedArray: assertUint8ClampedArray,
    undefined: assertUndefined,
    urlInstance: assertUrlInstance,
    urlSearchParams: assertUrlSearchParams,
    urlString: assertUrlString,
    validLength: assertValidLength,
    weakMap: assertWeakMap,
    weakRef: assertWeakRef,
    weakSet: assertWeakSet,
    whitespaceString: assertWhitespaceString,
};
const methodTypeMap = {
    isArray: 'Array',
    isArrayBuffer: 'ArrayBuffer',
    isArrayLike: 'array-like',
    isAsyncFunction: 'AsyncFunction',
    isAsyncGenerator: 'AsyncGenerator',
    isAsyncGeneratorFunction: 'AsyncGeneratorFunction',
    isAsyncIterable: 'AsyncIterable',
    isBigint: 'bigint',
    isBigInt64Array: 'BigInt64Array',
    isBigUint64Array: 'BigUint64Array',
    isBlob: 'Blob',
    isBoolean: 'boolean',
    isBoundFunction: 'Function',
    isBuffer: 'Buffer',
    isClass: 'Class',
    isDataView: 'DataView',
    isDate: 'Date',
    isDirectInstanceOf: 'T',
    /** @deprecated */
    isDomElement: 'HTMLElement',
    isEmptyArray: 'empty array',
    isEmptyMap: 'empty map',
    isEmptyObject: 'empty object',
    isEmptySet: 'empty set',
    isEmptyString: 'empty string',
    isEmptyStringOrWhitespace: 'empty string or whitespace',
    isEnumCase: 'EnumCase',
    isError: 'Error',
    isEvenInteger: 'even integer',
    isFalsy: 'falsy',
    isFloat32Array: 'Float32Array',
    isFloat64Array: 'Float64Array',
    isFormData: 'FormData',
    isFunction: 'Function',
    isGenerator: 'Generator',
    isGeneratorFunction: 'GeneratorFunction',
    isHtmlElement: 'HTMLElement',
    isInfinite: 'infinite number',
    isInRange: 'in range',
    isInt16Array: 'Int16Array',
    isInt32Array: 'Int32Array',
    isInt8Array: 'Int8Array',
    isInteger: 'integer',
    isIterable: 'Iterable',
    isMap: 'Map',
    isNan: 'NaN',
    isNativePromise: 'native Promise',
    isNegativeNumber: 'negative number',
    isNodeStream: 'Node.js Stream',
    isNonEmptyArray: 'non-empty array',
    isNonEmptyMap: 'non-empty map',
    isNonEmptyObject: 'non-empty object',
    isNonEmptySet: 'non-empty set',
    isNonEmptyString: 'non-empty string',
    isNonEmptyStringAndNotWhitespace: 'non-empty string and not whitespace',
    isNull: 'null',
    isNullOrUndefined: 'null or undefined',
    isNumber: 'number',
    isNumericString: 'string with a number',
    isObject: 'Object',
    isObservable: 'Observable',
    isOddInteger: 'odd integer',
    isPlainObject: 'plain object',
    isPositiveNumber: 'positive number',
    isPrimitive: 'primitive',
    isPromise: 'Promise',
    isPropertyKey: 'PropertyKey',
    isRegExp: 'RegExp',
    isSafeInteger: 'integer',
    isSet: 'Set',
    isSharedArrayBuffer: 'SharedArrayBuffer',
    isString: 'string',
    isSymbol: 'symbol',
    isTruthy: 'truthy',
    isTupleLike: 'tuple-like',
    isTypedArray: 'TypedArray',
    isUint16Array: 'Uint16Array',
    isUint32Array: 'Uint32Array',
    isUint8Array: 'Uint8Array',
    isUint8ClampedArray: 'Uint8ClampedArray',
    isUndefined: 'undefined',
    isUrlInstance: 'URL',
    isUrlSearchParams: 'URLSearchParams',
    isUrlString: 'string with a URL',
    isValidLength: 'valid length',
    isWeakMap: 'WeakMap',
    isWeakRef: 'WeakRef',
    isWeakSet: 'WeakSet',
    isWhitespaceString: 'whitespace string',
};
function keysOf(value) {
    return Object.keys(value);
}
const isMethodNames = keysOf(methodTypeMap);
function isIsMethodName(value) {
    return isMethodNames.includes(value);
}
export function assertAll(predicate, ...values) {
    if (!isAll(predicate, ...values)) {
        const expectedType = isIsMethodName(predicate.name) ? methodTypeMap[predicate.name] : 'predicate returns truthy for all values';
        throw new TypeError(typeErrorMessageMultipleValues(expectedType, values));
    }
}
export function assertAny(predicate, ...values) {
    if (!isAny(predicate, ...values)) {
        const predicates = isArray(predicate) ? predicate : [predicate];
        const expectedTypes = predicates.map(predicate => isIsMethodName(predicate.name) ? methodTypeMap[predicate.name] : 'predicate returns truthy for any value');
        throw new TypeError(typeErrorMessageMultipleValues(expectedTypes, values));
    }
}
export function assertArray(value, assertion) {
    if (!isArray(value)) {
        throw new TypeError(typeErrorMessage('Array', value));
    }
    if (assertion) {
        // eslint-disable-next-line unicorn/no-array-for-each, unicorn/no-array-callback-reference
        value.forEach(assertion);
    }
}
export function assertArrayBuffer(value) {
    if (!isArrayBuffer(value)) {
        throw new TypeError(typeErrorMessage('ArrayBuffer', value));
    }
}
export function assertArrayLike(value) {
    if (!isArrayLike(value)) {
        throw new TypeError(typeErrorMessage('array-like', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertAsyncFunction(value) {
    if (!isAsyncFunction(value)) {
        throw new TypeError(typeErrorMessage('AsyncFunction', value));
    }
}
export function assertAsyncGenerator(value) {
    if (!isAsyncGenerator(value)) {
        throw new TypeError(typeErrorMessage('AsyncGenerator', value));
    }
}
export function assertAsyncGeneratorFunction(value) {
    if (!isAsyncGeneratorFunction(value)) {
        throw new TypeError(typeErrorMessage('AsyncGeneratorFunction', value));
    }
}
export function assertAsyncIterable(value) {
    if (!isAsyncIterable(value)) {
        throw new TypeError(typeErrorMessage('AsyncIterable', value));
    }
}
export function assertBigint(value) {
    if (!isBigint(value)) {
        throw new TypeError(typeErrorMessage('bigint', value));
    }
}
export function assertBigInt64Array(value) {
    if (!isBigInt64Array(value)) {
        throw new TypeError(typeErrorMessage('BigInt64Array', value));
    }
}
export function assertBigUint64Array(value) {
    if (!isBigUint64Array(value)) {
        throw new TypeError(typeErrorMessage('BigUint64Array', value));
    }
}
export function assertBlob(value) {
    if (!isBlob(value)) {
        throw new TypeError(typeErrorMessage('Blob', value));
    }
}
export function assertBoolean(value) {
    if (!isBoolean(value)) {
        throw new TypeError(typeErrorMessage('boolean', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertBoundFunction(value) {
    if (!isBoundFunction(value)) {
        throw new TypeError(typeErrorMessage('Function', value));
    }
}
export function assertBuffer(value) {
    if (!isBuffer(value)) {
        throw new TypeError(typeErrorMessage('Buffer', value));
    }
}
export function assertClass(value) {
    if (!isClass(value)) {
        throw new TypeError(typeErrorMessage('Class', value));
    }
}
export function assertDataView(value) {
    if (!isDataView(value)) {
        throw new TypeError(typeErrorMessage('DataView', value));
    }
}
export function assertDate(value) {
    if (!isDate(value)) {
        throw new TypeError(typeErrorMessage('Date', value));
    }
}
export function assertDirectInstanceOf(instance, class_) {
    if (!isDirectInstanceOf(instance, class_)) {
        throw new TypeError(typeErrorMessage('T', instance));
    }
}
export function assertEmptyArray(value) {
    if (!isEmptyArray(value)) {
        throw new TypeError(typeErrorMessage('empty array', value));
    }
}
export function assertEmptyMap(value) {
    if (!isEmptyMap(value)) {
        throw new TypeError(typeErrorMessage('empty map', value));
    }
}
export function assertEmptyObject(value) {
    if (!isEmptyObject(value)) {
        throw new TypeError(typeErrorMessage('empty object', value));
    }
}
export function assertEmptySet(value) {
    if (!isEmptySet(value)) {
        throw new TypeError(typeErrorMessage('empty set', value));
    }
}
export function assertEmptyString(value) {
    if (!isEmptyString(value)) {
        throw new TypeError(typeErrorMessage('empty string', value));
    }
}
export function assertEmptyStringOrWhitespace(value) {
    if (!isEmptyStringOrWhitespace(value)) {
        throw new TypeError(typeErrorMessage('empty string or whitespace', value));
    }
}
export function assertEnumCase(value, targetEnum) {
    if (!isEnumCase(value, targetEnum)) {
        throw new TypeError(typeErrorMessage('EnumCase', value));
    }
}
export function assertError(value) {
    if (!isError(value)) {
        throw new TypeError(typeErrorMessage('Error', value));
    }
}
export function assertEvenInteger(value) {
    if (!isEvenInteger(value)) {
        throw new TypeError(typeErrorMessage('even integer', value));
    }
}
export function assertFalsy(value) {
    if (!isFalsy(value)) {
        throw new TypeError(typeErrorMessage('falsy', value));
    }
}
export function assertFloat32Array(value) {
    if (!isFloat32Array(value)) {
        throw new TypeError(typeErrorMessage('Float32Array', value));
    }
}
export function assertFloat64Array(value) {
    if (!isFloat64Array(value)) {
        throw new TypeError(typeErrorMessage('Float64Array', value));
    }
}
export function assertFormData(value) {
    if (!isFormData(value)) {
        throw new TypeError(typeErrorMessage('FormData', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertFunction(value) {
    if (!isFunction(value)) {
        throw new TypeError(typeErrorMessage('Function', value));
    }
}
export function assertGenerator(value) {
    if (!isGenerator(value)) {
        throw new TypeError(typeErrorMessage('Generator', value));
    }
}
export function assertGeneratorFunction(value) {
    if (!isGeneratorFunction(value)) {
        throw new TypeError(typeErrorMessage('GeneratorFunction', value));
    }
}
export function assertHtmlElement(value) {
    if (!isHtmlElement(value)) {
        throw new TypeError(typeErrorMessage('HTMLElement', value));
    }
}
export function assertInfinite(value) {
    if (!isInfinite(value)) {
        throw new TypeError(typeErrorMessage('infinite number', value));
    }
}
export function assertInRange(value, range) {
    if (!isInRange(value, range)) {
        throw new TypeError(typeErrorMessage('in range', value));
    }
}
export function assertInt16Array(value) {
    if (!isInt16Array(value)) {
        throw new TypeError(typeErrorMessage('Int16Array', value));
    }
}
export function assertInt32Array(value) {
    if (!isInt32Array(value)) {
        throw new TypeError(typeErrorMessage('Int32Array', value));
    }
}
export function assertInt8Array(value) {
    if (!isInt8Array(value)) {
        throw new TypeError(typeErrorMessage('Int8Array', value));
    }
}
export function assertInteger(value) {
    if (!isInteger(value)) {
        throw new TypeError(typeErrorMessage('integer', value));
    }
}
export function assertIterable(value) {
    if (!isIterable(value)) {
        throw new TypeError(typeErrorMessage('Iterable', value));
    }
}
export function assertMap(value) {
    if (!isMap(value)) {
        throw new TypeError(typeErrorMessage('Map', value));
    }
}
export function assertNan(value) {
    if (!isNan(value)) {
        throw new TypeError(typeErrorMessage('NaN', value));
    }
}
export function assertNativePromise(value) {
    if (!isNativePromise(value)) {
        throw new TypeError(typeErrorMessage('native Promise', value));
    }
}
export function assertNegativeNumber(value) {
    if (!isNegativeNumber(value)) {
        throw new TypeError(typeErrorMessage('negative number', value));
    }
}
export function assertNodeStream(value) {
    if (!isNodeStream(value)) {
        throw new TypeError(typeErrorMessage('Node.js Stream', value));
    }
}
export function assertNonEmptyArray(value) {
    if (!isNonEmptyArray(value)) {
        throw new TypeError(typeErrorMessage('non-empty array', value));
    }
}
export function assertNonEmptyMap(value) {
    if (!isNonEmptyMap(value)) {
        throw new TypeError(typeErrorMessage('non-empty map', value));
    }
}
export function assertNonEmptyObject(value) {
    if (!isNonEmptyObject(value)) {
        throw new TypeError(typeErrorMessage('non-empty object', value));
    }
}
export function assertNonEmptySet(value) {
    if (!isNonEmptySet(value)) {
        throw new TypeError(typeErrorMessage('non-empty set', value));
    }
}
export function assertNonEmptyString(value) {
    if (!isNonEmptyString(value)) {
        throw new TypeError(typeErrorMessage('non-empty string', value));
    }
}
export function assertNonEmptyStringAndNotWhitespace(value) {
    if (!isNonEmptyStringAndNotWhitespace(value)) {
        throw new TypeError(typeErrorMessage('non-empty string and not whitespace', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertNull(value) {
    if (!isNull(value)) {
        throw new TypeError(typeErrorMessage('null', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertNullOrUndefined(value) {
    if (!isNullOrUndefined(value)) {
        throw new TypeError(typeErrorMessage('null or undefined', value));
    }
}
export function assertNumber(value) {
    if (!isNumber(value)) {
        throw new TypeError(typeErrorMessage('number', value));
    }
}
export function assertNumericString(value) {
    if (!isNumericString(value)) {
        throw new TypeError(typeErrorMessage('string with a number', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertObject(value) {
    if (!isObject(value)) {
        throw new TypeError(typeErrorMessage('Object', value));
    }
}
export function assertObservable(value) {
    if (!isObservable(value)) {
        throw new TypeError(typeErrorMessage('Observable', value));
    }
}
export function assertOddInteger(value) {
    if (!isOddInteger(value)) {
        throw new TypeError(typeErrorMessage('odd integer', value));
    }
}
export function assertPlainObject(value) {
    if (!isPlainObject(value)) {
        throw new TypeError(typeErrorMessage('plain object', value));
    }
}
export function assertPositiveNumber(value) {
    if (!isPositiveNumber(value)) {
        throw new TypeError(typeErrorMessage('positive number', value));
    }
}
export function assertPrimitive(value) {
    if (!isPrimitive(value)) {
        throw new TypeError(typeErrorMessage('primitive', value));
    }
}
export function assertPromise(value) {
    if (!isPromise(value)) {
        throw new TypeError(typeErrorMessage('Promise', value));
    }
}
export function assertPropertyKey(value) {
    if (!isPropertyKey(value)) {
        throw new TypeError(typeErrorMessage('PropertyKey', value));
    }
}
export function assertRegExp(value) {
    if (!isRegExp(value)) {
        throw new TypeError(typeErrorMessage('RegExp', value));
    }
}
export function assertSafeInteger(value) {
    if (!isSafeInteger(value)) {
        throw new TypeError(typeErrorMessage('integer', value));
    }
}
export function assertSet(value) {
    if (!isSet(value)) {
        throw new TypeError(typeErrorMessage('Set', value));
    }
}
export function assertSharedArrayBuffer(value) {
    if (!isSharedArrayBuffer(value)) {
        throw new TypeError(typeErrorMessage('SharedArrayBuffer', value));
    }
}
export function assertString(value) {
    if (!isString(value)) {
        throw new TypeError(typeErrorMessage('string', value));
    }
}
export function assertSymbol(value) {
    if (!isSymbol(value)) {
        throw new TypeError(typeErrorMessage('symbol', value));
    }
}
export function assertTruthy(value) {
    if (!isTruthy(value)) {
        throw new TypeError(typeErrorMessage('truthy', value));
    }
}
export function assertTupleLike(value, guards) {
    if (!isTupleLike(value, guards)) {
        throw new TypeError(typeErrorMessage('tuple-like', value));
    }
}
export function assertTypedArray(value) {
    if (!isTypedArray(value)) {
        throw new TypeError(typeErrorMessage('TypedArray', value));
    }
}
export function assertUint16Array(value) {
    if (!isUint16Array(value)) {
        throw new TypeError(typeErrorMessage('Uint16Array', value));
    }
}
export function assertUint32Array(value) {
    if (!isUint32Array(value)) {
        throw new TypeError(typeErrorMessage('Uint32Array', value));
    }
}
export function assertUint8Array(value) {
    if (!isUint8Array(value)) {
        throw new TypeError(typeErrorMessage('Uint8Array', value));
    }
}
export function assertUint8ClampedArray(value) {
    if (!isUint8ClampedArray(value)) {
        throw new TypeError(typeErrorMessage('Uint8ClampedArray', value));
    }
}
export function assertUndefined(value) {
    if (!isUndefined(value)) {
        throw new TypeError(typeErrorMessage('undefined', value));
    }
}
export function assertUrlInstance(value) {
    if (!isUrlInstance(value)) {
        throw new TypeError(typeErrorMessage('URL', value));
    }
}
// eslint-disable-next-line unicorn/prevent-abbreviations
export function assertUrlSearchParams(value) {
    if (!isUrlSearchParams(value)) {
        throw new TypeError(typeErrorMessage('URLSearchParams', value));
    }
}
export function assertUrlString(value) {
    if (!isUrlString(value)) {
        throw new TypeError(typeErrorMessage('string with a URL', value));
    }
}
export function assertValidLength(value) {
    if (!isValidLength(value)) {
        throw new TypeError(typeErrorMessage('valid length', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertWeakMap(value) {
    if (!isWeakMap(value)) {
        throw new TypeError(typeErrorMessage('WeakMap', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertWeakRef(value) {
    if (!isWeakRef(value)) {
        throw new TypeError(typeErrorMessage('WeakRef', value));
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types
export function assertWeakSet(value) {
    if (!isWeakSet(value)) {
        throw new TypeError(typeErrorMessage('WeakSet', value));
    }
}
export function assertWhitespaceString(value) {
    if (!isWhitespaceString(value)) {
        throw new TypeError(typeErrorMessage('whitespace string', value));
    }
}
export default is;
